/**
 *
 * @file dpocon.c
 *
 *  PLASMA computational routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.5.0
 * @author Jakub Kurzak
 * @date 2012-10-04
 * @generated d Thu Nov  8 11:44:36 2012
 *
 **/
#include <lapacke.h>
#include "common.h"
#undef COMPLEX
#define REAL
/* Temporary declaration of dlacn2 */
int LAPACK_GLOBAL(dlacn2, DLACN2)( int *N, double *V, double *X,
#if !defined(COMPLEX)
                                   int *ISGN,
#endif
                                   double *EST, int *KASE, int *ISAVE);

/***************************************************************************//**
 *
 * @ingroup double
 *
 *  PLASMA_dpocon - estimates the reciprocal of the condition number (in the
 *  1-norm) of a complex Hermitian positive definite matrix using the
 *  Cholesky factorization A = U**T*U or A = L*L**T computed by DPOTRF.
 *
 *  An estimate is obtained for norm(inv(A)), and the reciprocal of the
 *  condition number is computed as RCOND = 1 / (ANORM * norm(inv(A))).
 *
 *******************************************************************************
 *
 * @param[in] uplo
 *          = PlasmaUpper: Upper triangle of A is stored;
 *          = PlasmaLower: Lower triangle of A is stored.
 *
 * @param[in] N
 *          The order of the matrix A. N >= 0.
 *
 * @param[in] A
 *          The N-by-N matrix A.
 *
 * @param[in] LDA
 *          The leading dimension of the array A. LDA >= max(1,N).
 *
 * @param[in] anorm
 *          If norm = PlasmaOneNorm, the 1-norm of the original matrix A.
 *          If norm = PlasmaInfNorm, the infinity-norm of the original matrix A.
 *
 * \param[out] rcond
 *          The reciprocal of the condition number of the matrix A,
 *          computed as stated above.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *          \retval <0 if -i, the i-th argument had an illegal value
 *
 *******************************************************************************
 *
 * @sa PLASMA_dpocon_Tile
 * @sa PLASMA_dpocon_Tile_Async
 * @sa PLASMA_cpocon
 * @sa PLASMA_dpocon
 * @sa PLASMA_spocon
 *
 ******************************************************************************/
int PLASMA_dpocon(PLASMA_enum uplo, int N,
                  double *A, int LDA, double anorm, double *rcond)
{
    int NB;
    int status;
    plasma_context_t *plasma;
    PLASMA_sequence *sequence = NULL;
    PLASMA_request request = PLASMA_REQUEST_INITIALIZER;
    PLASMA_desc descA;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_dpocon", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */

    /* Quick return */

    /* Tune NB depending on M, N & NRHS; Set NBNB */
    status = plasma_tune(PLASMA_FUNC_DGESV, N, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dpocon", "plasma_tune() failed");
        return status;
    }
    /* Set NT */
    NB = PLASMA_NB;
    plasma_sequence_create(plasma, &sequence);

    if (PLASMA_TRANSLATION == PLASMA_OUTOFPLACE) {
        plasma_dooplap2tile(
            descA, A, NB, NB, LDA, N, 0, 0, N, N,
            sequence, &request, plasma_desc_mat_free(&(descA)));
    } else {
        plasma_diplap2tile(
            descA, A, NB, NB, LDA, N, 0, 0, N, N, sequence, &request);
    }

    /* Call the tile interface */
    PLASMA_dpocon_Tile_Async(uplo, &descA, anorm, rcond, sequence, &request);

    if (PLASMA_TRANSLATION == PLASMA_OUTOFPLACE) {
        plasma_dynamic_sync();
        plasma_desc_mat_free(&descA);
    } else {
        plasma_diptile2lap(descA, A, NB, NB, LDA, N, sequence, &request);
        plasma_dynamic_sync();
    }

    status = sequence->status;
    plasma_sequence_destroy(plasma, sequence);
    return status;
}

/***************************************************************************//**
 *
 * @ingroup double_Tile
 *
 *  PLASMA_dpocon_Tile - estimates the reciprocal of the condition number
 *  of a general complex matrix A, in either the 1-norm or the infinity-norm.
 *  Tile equivalent of PLASMA_dpocon().
 *  Operates on matrices stored by tiles.
 *  All matrices are passed through descriptors.
 *  All dimensions are taken from the descriptors.
 *
 *******************************************************************************
 *
 * @param[in] norm
 *          Specifies whether the 1-norm condition number
 *          or the infinity-norm condition number is required:
 *          = PlasmaOneNorm: One norm
 *          = PlasmaInfNorm: Infinity norm
 *
 * @param[in] A
 *          The N-by-N matrix A.
 *
 * @param[in] anorm
 *          If norm = PlasmaOneNorm, the 1-norm of the original matrix A.
 *          If norm = PlasmaInfNorm, the infinity-norm of the original matrix A.
 *
 * \param[out] rcond
 *          The reciprocal of the condition number of the matrix A,
 *          computed as stated above.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *          \retval <0 if -i, the i-th argument had an illegal value
 *
 *******************************************************************************
 *
 * @sa PLASMA_dpocon
 * @sa PLASMA_dpocon_Tile_Async
 * @sa PLASMA_cpocon_Tile
 * @sa PLASMA_dpocon_Tile
 * @sa PLASMA_spocon_Tile
 *
 ******************************************************************************/
int PLASMA_dpocon_Tile(PLASMA_enum uplo, PLASMA_desc *A, double anorm, double *rcond)
{
    plasma_context_t *plasma;
    PLASMA_sequence *sequence = NULL;
    PLASMA_request request = PLASMA_REQUEST_INITIALIZER;
    int status;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_dpocon_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    plasma_sequence_create(plasma, &sequence);
    PLASMA_dpocon_Tile_Async(uplo, A, anorm, rcond, sequence, &request);
    plasma_dynamic_sync();
    status = sequence->status;
    plasma_sequence_destroy(plasma, sequence);
    return status;
}

/***************************************************************************//**
 *
 * @ingroup double_Tile_Async
 *
 *  PLASMA_dpocon_Tile_Async - estimates the reciprocal of the condition number
 *  of a general complex matrix A, in either the 1-norm or the infinity-norm.
 *  Non-blocking equivalent of PLASMA_dpocon_Tile().
 *  May return before the computation is finished.
 *  Allows for pipelining of operations at runtime.
 *
 *******************************************************************************
 *
 * @param[in] sequence
 *          Identifies the sequence of function calls that this call belongs to
 *          (for completion checks and exception handling purposes).
 *
 * @param[out] request
 *          Identifies this function call (for exception handling purposes).
 *
 *******************************************************************************
 *
 * @sa PLASMA_dpocon
 * @sa PLASMA_dpocon_Tile
 * @sa PLASMA_cpocon_Tile_Async
 * @sa PLASMA_dpocon_Tile_Async
 * @sa PLASMA_spocon_Tile_Async
 *
 ******************************************************************************/
int PLASMA_dpocon_Tile_Async(PLASMA_enum uplo, PLASMA_desc *A, double anorm, double *rcond,
                             PLASMA_sequence *sequence, PLASMA_request *request)
{
    PLASMA_desc descA;
    plasma_context_t *plasma;
    int kase, kase1;
    double ainvnm;
    int isave[3], itrs = 0;
    int fallback = PLASMA_FALSE;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_dpocon_Tile_Async", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    if (sequence == NULL) {
        plasma_fatal_error("PLASMA_dpocon_Tile_Async", "NULL sequence");
        return PLASMA_ERR_UNALLOCATED;
    }
    if (request == NULL) {
        plasma_fatal_error("PLASMA_dpocon_Tile_Async", "NULL request");
        return PLASMA_ERR_UNALLOCATED;
    }
    /* Check sequence status */
    if (sequence->status == PLASMA_SUCCESS)
        request->status = PLASMA_SUCCESS;
    else
        return plasma_request_fail(sequence, request, PLASMA_ERR_SEQUENCE_FLUSHED);

    /* Check descriptors for correctness */
    if (plasma_desc_check(A) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dpocon_Tile_Async", "invalid first descriptor");
        return plasma_request_fail(sequence, request, PLASMA_ERR_ILLEGAL_VALUE);
    } else {
        descA = *A;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb) {
        plasma_error("PLASMA_dpocon_Tile_Async", "only square tiles supported");
        return plasma_request_fail(sequence, request, PLASMA_ERR_ILLEGAL_VALUE);
    }
    /* Quick return */

    /* Estimate the norm of inv(A). */
    ainvnm = 0.0;
    kase = 0;

    double *work  = (double*)plasma_shared_alloc(plasma, descA.m, PlasmaRealDouble);
    double *workN = (double*)plasma_shared_alloc(plasma, descA.m, PlasmaRealDouble);
#if defined(REAL) 
    int *isgn = (int*)plasma_shared_alloc(plasma, descA.m, PlasmaInteger);
#endif

    PLASMA_desc desc_work = plasma_desc_init(
        PlasmaRealDouble, descA.nb, descA.nb, descA.bsiz, descA.m, 1, 0, 0, descA.m, 1);
    desc_work.mat = work;

    do {
        itrs ++;
#if defined(REAL)
        LAPACK_GLOBAL(dlacn2, DLACN2)(&descA.m, workN, desc_work.mat, isgn, &ainvnm, &kase, isave);
                           /* dlacn2_(&descA.m, workN, desc_work.mat, isgn, &ainvnm, &kase, isave); */
#else
        LAPACK_GLOBAL(dlacn2, DLACN2)(&descA.m, workN, desc_work.mat, &ainvnm, &kase, isave);
                           /* dlacn2_(&descA.m, workN, desc_work.mat, &ainvnm, &kase, isave); */
#endif
#define FALLBACK
#ifdef  FALLBACK
        /* printf( " %d: ainvnm=%.2e\n",itrs,ainvnm ); */
        if( isnan(ainvnm) || isinf(ainvnm) || ainvnm > LAPACKE_dlamch('O') ) {
            /* fall back to LAPACK */
            /*printf( " ainvnm=%.2e, fallback to LAPACK\n",ainvnm );  */
            int info, m = descA.m, n = descA.n, nb = descA.nb;
            double *Atmp = (double*)malloc(m*n*sizeof(double));

            plasma_dooptile2lap( descA, Atmp, nb, nb, m, n, sequence, request);
            if( uplo == PlasmaLower ) {
                info = LAPACKE_dpocon(LAPACK_COL_MAJOR, 'L', n, Atmp, m, anorm, rcond);
            } else {
                info = LAPACKE_dpocon(LAPACK_COL_MAJOR, 'U', n, Atmp, m, anorm, rcond);
            } 
            free(Atmp);
            fallback = PLASMA_TRUE;
            sequence->status = info;
            kase = 0;
        }
#endif
        if (kase != 0) {
            if (uplo == PlasmaLower) {
                /* Multiply by inv(L). */
                plasma_parallel_call_9(plasma_pdtrsm,
                    PLASMA_enum, PlasmaLeft,
                    PLASMA_enum, PlasmaLower,
                    PLASMA_enum, PlasmaNoTrans,
                    PLASMA_enum, PlasmaNonUnit,
                    double, 1.0,
                    PLASMA_desc, descA,
                    PLASMA_desc, desc_work,
                    PLASMA_sequence*, sequence,
                    PLASMA_request*, request);

                /* Multiply by inv(U). */
                plasma_parallel_call_9(plasma_pdtrsm,
                    PLASMA_enum, PlasmaLeft,
                    PLASMA_enum, PlasmaLower,
                    PLASMA_enum, PlasmaTrans,
                    PLASMA_enum, PlasmaNonUnit,
                    double, 1.0,
                    PLASMA_desc, descA,
                    PLASMA_desc, desc_work,
                    PLASMA_sequence*, sequence,
                    PLASMA_request*, request);
            }
            else {
                /* Multiply by inv(U**T). */
                plasma_parallel_call_9(plasma_pdtrsm,
                    PLASMA_enum, PlasmaLeft,
                    PLASMA_enum, PlasmaUpper,
                    PLASMA_enum, PlasmaTrans,
                    PLASMA_enum, PlasmaNonUnit,
                    double, 1.0,
                    PLASMA_desc, descA,
                    PLASMA_desc, desc_work,
                    PLASMA_sequence*, sequence,
                    PLASMA_request*, request);

                /* Multiply by inv(L**T). */
                plasma_parallel_call_9(plasma_pdtrsm,
                    PLASMA_enum, PlasmaLeft,
                    PLASMA_enum, PlasmaUpper,
                    PLASMA_enum, PlasmaNoTrans,
                    PLASMA_enum, PlasmaNonUnit,
                    double, 1.0,
                    PLASMA_desc, descA,
                    PLASMA_desc, desc_work,
                    PLASMA_sequence*, sequence,
                    PLASMA_request*, request);
            }
        }
        plasma_dynamic_sync();
    }
    while (kase != 0);

    /* Compute the estimate of the reciprocal condition number. */
    if (ainvnm != 0.0 && !fallback)
        *rcond = (1.0 / ainvnm) / anorm;

#if defined(REAL)
    plasma_shared_free(plasma, isgn);
#endif
    plasma_shared_free(plasma, work);
    plasma_shared_free(plasma, workN);

    return PLASMA_SUCCESS;
}
