/**
 * \file splgsy.c
 *
 *  PLASMA computational routines
 *  Release Date: November, 15th 2009
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.3.0
 * @author Mathieu Faverge
 * @date 2010-11-15
 * @generated s
 *
 **/
#if defined( _WIN32 ) || defined( _WIN64 )
typedef __int64 int64_t;
#else 
#  include <inttypes.h>
#endif
#include <plasma.h>
#include <lapacke.h>
#include <math.h>

void *plasma_getaddr(PLASMA_desc A, int m, int n);
#define BLKADDR(A, m, n)  ((float *)plasma_getaddr(*A, m, n))
#define BLKLDD(A, k) ( ( (k) + (A).i/(A).mb) < (A).lm1 ? (A).mb : (A).lm%(A).mb )
//#define AIJ(A, i, j) (&(((float*)(A->mat))[(int64_t)(j)*bsiz*(int64_t)(A->lmt) + (int64_t)(i)*bsiz ]))

 /** ****************************************************************************
 *
 * @ingroup float
 *
 *  PLASMA_splgsy - Generate a hermitian matrice with diagonal dominant.
 *
 *******************************************************************************
 *
 * @param[in] N
 *          The number of rows/columns of A. N >= 0.
 *
 * @param[out] A
 *          On entry, the N-by-N uninitialized matrix A.
 *          On exit, an N-by-N hermitian matrix with diagonal
 *          dominant.
 *
 * @param[in] LDA
 *          The leading dimension of the array A. LDA >= max(1,N).
 *
 * @param[in,out] ISEED
 *          Array of dimension 4.
 *          On entry, the seed of the random number generator; the
 *          array elements must be between 0 and 4095, and ISEED(4)
 *          must be odd.
 *          On exit, the seed is updated.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 *******************************************************************************
 *
 * @sa PLASMA_splgsy_Tile
 * @sa PLASMA_cplghe
 * @sa PLASMA_dplghe
 * @sa PLASMA_splghe
 *
 ******************************************************************************/
int PLASMA_splgsy(int N, float *A, int LDA, int *ISEED )
{
    int i, j;
     
    for ( i=0; i<N; i++) {
        /* Initialize column */
        LAPACKE_slarnv_work(1, ISEED, N-i, &A[i*LDA+i]);
        /* Copy uguate in row */
        for ( j=i; j<N; j++)
            A[ LDA*j + i ] = (A[ LDA*i + j]);
        
        A[ LDA*i+i ] += (float)N ;
    }

    return PLASMA_SUCCESS;
}

 /** ****************************************************************************
 *
 * @ingroup float
 *
 *  PLASMA_splgsy_Tile - Generate a hermitian matrice in tile layout
 *  with diagonal dominant.
 *
 *******************************************************************************
 *
 * @param[in,out] A
 *          The descriptor on the matrix to initialize.
 *          On exit, the matrix dscribe by the descriptor is hermitian
 *          diagonal dominant.
 *
 * @param[in] LDA
 *          The leading dimension of the array A. LDA >= max(1,N).
 *
 * @param[in,out] ISEED
 *          Array of dimension 4.
 *          On entry, the seed of the random number generator; the
 *          array elements must be between 0 and 4095, and ISEED(4)
 *          must be odd.
 *          On exit, the seed is updated.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 *******************************************************************************
 *
 * @sa PLASMA_splgsy_Tile
 * @sa PLASMA_cplghe
 * @sa PLASMA_dplghe
 * @sa PLASMA_splghe
 *
 ******************************************************************************/
int PLASMA_splgsy_Tile(PLASMA_desc *A, int *ISEED )
{
    int i, j, it, jt;
    int n    = A->ln;
    int nt   = A->mt;
    int nb   = A->nb;
    int64_t bsiz = A->bsiz;

    for ( it = 0; it < nt; it++) {
        for ( jt = it; jt < nt; jt++) {
            /* Initialize block extra-diagonal */
            LAPACKE_slarnv_work(1, ISEED, bsiz, BLKADDR(A, it, jt));
            
            /* Conjuguate the opposite block */
            for (i = 0; i < nb; i++)
                for (j = 0; j < nb; j++)
                    BLKADDR(A, jt, it)[ j*nb + i] = (BLKADDR(A, it, jt)[ i*nb + j]);
        }

        /* Initialize block diagonal */
        for ( i=0; i<nb; i++) {
            /* Initialize column */
            LAPACKE_slarnv_work(1, ISEED, nb-i, (void *)&(BLKADDR(A, it, it)[i*nb+i]));
            /* Copy uguate in row */
            for ( j=i; j<nb; j++)
                BLKADDR( A, it, it)[ nb*j + i ] = (BLKADDR(A, it, it)[ nb*i + j ]);
            BLKADDR(A, it, it)[i*nb+i] += (float)n ;
        }
    }

    return PLASMA_SUCCESS;
}
