/**
 *
 * @file common.h
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.3.0
 * @author Jakub Kurzak
 * @date 2010-11-15
 *
 **/

/***************************************************************************//**
 *  PLASMA facilities of interest to both PLASMA core developer
 *  and also of interest to PLASMA community contributor.
 **/
#ifndef _PLASMA_COMMON_H_
#define _PLASMA_COMMON_H_

#if defined( _WIN32 ) || defined( _WIN64 )
#include <io.h>
#else
#include <unistd.h>
#endif

#include "global.h"
#include "context.h"
#include "control.h"
#include "core_blas.h"
#include "allocate.h"
#include "auxiliary.h"
#include "descriptor.h"
#include "tile.h"
#include "async.h"

/***************************************************************************//**
 *  Global shortcuts
 **/
#define PLASMA_RANK        plasma_rank(plasma)
#define PLASMA_SIZE        plasma->world_size
#define PLASMA_GRPSIZE     plasma->group_size
#define PLASMA_NB          plasma->nb
#define PLASMA_IB          plasma->ib
#define PLASMA_NBNBSIZE    plasma->nbnbsize
#define PLASMA_IBNBSIZE    plasma->ibnbsize
#define PLASMA_INFO        plasma->info
#define PLASMA_SCHEDULING  plasma->scheduling
#define PLASMA_RHBLK       plasma->rhblock
#define PLASMA_TRANSLATION plasma->translation

/***************************************************************************//**
 *  IPT internal define
 **/
#define PlasmaIPT_NoDep   0
#define PlasmaIPT_Panel   1
#define PlasmaIPT_All     2

/***************************************************************************//**
 *  Global utilities
 **/
#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif
#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif
#ifndef roundup
#define roundup(a, b) (b <= 0) ? (a) : (((a) + (b)-1) & ~((b)-1))
#endif

/***************************************************************************//**
 *  Static scheduler
 **/
#define ss_init(m, n, init_val) \
{ \
    if (PLASMA_RANK == 0) { \
        plasma->ss_progress = (volatile int *)plasma_shared_alloc(plasma, (m)*(n), PlasmaInteger); \
        plasma_memset_int((int*)plasma->ss_progress, (m)*(n), (init_val)); \
    } \
    plasma->ss_ld = (m); \
    plasma->ss_abort = 0; \
    plasma_barrier(plasma); \
}

#define ss_abort()   plasma->ss_abort = 1;
#define ss_aborted() plasma->ss_abort

#define ss_cond_set(m, n, val) \
{ \
    plasma->ss_progress[(m)+plasma->ss_ld*(n)] = (val); \
}

#define ss_cond_wait(m, n, val) \
{ \
    while (!plasma->ss_abort && \
            plasma->ss_progress[(m)+plasma->ss_ld*(n)] != (val)) \
        plasma_yield(); \
    if (plasma->ss_abort) \
        break; \
}

#define ss_finalize() \
{ \
    plasma_barrier(plasma); \
    if (PLASMA_RANK == 0) \
        plasma_shared_free(plasma, (void*)plasma->ss_progress); \
}

/***************************************************************************//**
 *  Global array of LAPACK constants
 **/
extern char *plasma_lapack_constants[];

#ifdef __cplusplus
extern "C" {
#endif

#include "compute_s.h"
#include "compute_d.h"
#define COMPLEX
#include "compute_c.h"
#include "compute_z.h"
#undef COMPLEX

void plasma_pdlag2s(plasma_context_t *plasma);
void plasma_pzlag2c(plasma_context_t *plasma);
void plasma_pslag2d(plasma_context_t *plasma);
void plasma_pclag2z(plasma_context_t *plasma);

#ifdef __cplusplus
}
#endif

#endif
