/**
 * \file cplghe.c
 *
 *  PLASMA computational routines
 *  Release Date: November, 15th 2009
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Mathieu Faverge
 * @date 2009-11-15
 *
 **/
#include <inttypes.h>
#include <lapack.h>
#include <math.h>
#include <plasma.h>

#define AIJ(A, i, j) (&(((PLASMA_Complex32_t*)(A->mat))[(int64_t)(j)*bsiz*(int64_t)(A->lmt) + (int64_t)(i)*bsiz ]))

 /** ****************************************************************************
 *
 * @ingroup PLASMA_Complex32_t
 *
 *  PLASMA_cplghe - Generate a hermitian matrice with diagonal dominant.
 *
 *******************************************************************************
 *
 * @param[in] N
 *          The number of rows/columns of A. N >= 0.
 *
 * @param[out] A
 *          On entry, the N-by-N uninitialized matrix A.
 *          On exit, an N-by-N hermitian matrix with diagonal
 *          dominant.
 *
 * @param[in] LDA
 *          The leading dimension of the array A. LDA >= max(1,N).
 *
 * @param[in,out] ISEED
 *          Array of dimension 4.
 *          On entry, the seed of the random number generator; the
 *          array elements must be between 0 and 4095, and ISEED(4)
 *          must be odd.
 *          On exit, the seed is updated.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 *******************************************************************************
 *
 * @sa PLASMA_cplghe_Tile
 * @sa PLASMA_zplghe
 * @sa PLASMA_dplghe
 * @sa PLASMA_splghe
 *
 ******************************************************************************/
int PLASMA_cplghe(int N, PLASMA_Complex32_t *A, int LDA, int *ISEED )
{
    int i, j;
     
    for ( i=0; i<N; i++) {
        /* Initialize column */
        lapack_clarnv(1, ISEED, N-i, &A[i*LDA+i]);
        /* Copy conjfuguate in row */
        for ( j=i; j<N; j++)
            A[ LDA*j + i ] = conjf(A[ LDA*i + j]);
        
        A[ LDA*i+i ] += (PLASMA_Complex32_t)N ;
    }

    return PLASMA_SUCCESS;
}

 /** ****************************************************************************
 *
 * @ingroup PLASMA_Complex32_t
 *
 *  PLASMA_cplghe_Tile - Generate a hermitian matrice in tile layout
 *  with diagonal dominant.
 *
 *******************************************************************************
 *
 * @param[in,out] A
 *          The descriptor on the matrix to initialize.
 *          On exit, the matrix dscribe by the descriptor is hermitian
 *          diagonal dominant.
 *
 * @param[in] LDA
 *          The leading dimension of the array A. LDA >= max(1,N).
 *
 * @param[in,out] ISEED
 *          Array of dimension 4.
 *          On entry, the seed of the random number generator; the
 *          array elements must be between 0 and 4095, and ISEED(4)
 *          must be odd.
 *          On exit, the seed is updated.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 *******************************************************************************
 *
 * @sa PLASMA_cplghe_Tile
 * @sa PLASMA_zplghe
 * @sa PLASMA_dplghe
 * @sa PLASMA_splghe
 *
 ******************************************************************************/
int PLASMA_cplghe_Tile(PLASMA_desc *A, int *ISEED )
{
    int i, j, it, jt;
    int n    = A->ln;
    int nt   = A->mt;
    int nb   = A->nb;
    int64_t bsiz = A->bsiz;

    for ( it = 0; it < nt; it++) {
        for ( jt = it; jt < nt; jt++) {
            /* Initialize block extra-diagonal */
            lapack_clarnv(1, ISEED, bsiz, AIJ(A, it, jt));
            
            /* Conjuguate the opposite block */
            for (i = 0; i < nb; i++)
                for (j = 0; j < nb; j++)
                    AIJ(A, jt, it)[ j*nb + i] = conjf(AIJ(A, it, jt)[ i*nb + j]);
        }

        /* Initialize block diagonal */
        for ( i=0; i<nb; i++) {
            /* Initialize column */
            lapack_clarnv(1, ISEED, nb-i, (void *)&(AIJ(A, it, it)[i*nb+i]));
            /* Copy conjfuguate in row */
            for ( j=i; j<nb; j++)
                AIJ( A, it, it)[ nb*j + i ] = conjf(AIJ(A, it, it)[ nb*i + j ]);
            AIJ(A, it, it)[i*nb+i] += (PLASMA_Complex32_t)n ;
        }
    }

    return PLASMA_SUCCESS;
}
