/**
 *
 * @file testing_dsungesv.c
 *
 *  PLASMA testing routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Emmanuel Agullo
 * @date 2009-11-15
 *
 **/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <cblas.h>
#include <lapack.h>
#include <plasma.h>

#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif
#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif

int check_solution(int, int, double*, int, double*, double*, int, double);

int IONE=1;
int ISEED[4] = {0,0,0,1};   /* initial seed for dlarnv() */

int main (int argc, char **argv)
{
    /* Check for number of arguments*/
    if ( argc != 6 ){
        printf(" Proper Usage is : ./testing_dsgels ncores N LDA NRHS LDB with \n"
               "   - ncores : number of cores \n"
               "   - N    : size of the matrix A\n"
               "   - LDA  : leading dimension of the matrix A \n"
               "   - NRHS : number of RHS \n"
               "   - LDB  : leading dimension of the matrix B\n");
        exit(1);
    }

    int cores = atoi(argv[1]);
    int N     = atoi(argv[2]);
    int LDA   = atoi(argv[3]);
    int NRHS  = atoi(argv[4]);
    int LDB   = atoi(argv[5]);
    int LDX   = LDB;
    int ITER;

    double eps;
    int info_solution;
    int i,j;
    int LDAxN    = LDA*N;
    int LDBxNRHS = LDB*NRHS;

    double *A1 = (double *)malloc(LDA*N   *sizeof(double));
    double *A2 = (double *)malloc(LDA*N   *sizeof(double));
    double *B1 = (double *)malloc(LDB*NRHS*sizeof(double));
    double *B2 = (double *)malloc(LDB*NRHS*sizeof(double));
    double *X  = (double *)malloc(LDX*NRHS*sizeof(double));

    /* Check if unable to allocate memory */
    if ( (!A1) || (!A2) || (!B1) || (!B2) || (!X) ) {
        printf("Out of Memory \n ");
        exit(0);
    }

    /* Plasma Initialization */
    PLASMA_Init(cores);

    /*
    PLASMA_Disable(PLASMA_AUTOTUNING);
    PLASMA_Set(PLASMA_TILE_SIZE, 6);
    PLASMA_Set(PLASMA_INNER_BLOCK_SIZE, 3);
    */

    eps = lapack_dlamch(lapack_eps);

    /*----------------------------------------------------------
    *  TESTING DSGELS
    */

    /* Initialize A1 and A2 */
    lapack_dlarnv(IONE, ISEED, LDAxN, A1);
    for (i = 0; i < N; i++)
        for (j = 0; j < N; j++)
            A2[LDA*j+i] = A1[LDA*j+i] ;

    /* Initialize B1 and B2 */
    lapack_dlarnv(IONE, ISEED, LDBxNRHS, B1);
    for (i = 0; i < N; i++)
        for (j = 0; j < NRHS; j++)
             B2[LDB*j+i] = B1[LDB*j+i] ;

    printf("\n");
    printf("------ TESTS FOR PLASMA DSUNGESV ROUTINE -------  \n");
    printf("            Size of the Matrix %d by %d\n", N, N);
    printf("\n");
    printf(" The matrix A is randomly generated for each test.\n");
    printf("============\n");
    printf(" The relative machine precision (eps) is to be %e \n",eps);
    printf(" Computational tests pass if scaled residuals are less than 60.\n");

    /* PLASMA DSUNGESV */
    PLASMA_dsungesv(PlasmaNoTrans, N, NRHS, A2, LDA, B2, LDB, X, LDX, &ITER);

    printf("\n");
    printf(" Solution obtained with %d iterations\n", ITER);
    printf("\n");
   
    /* Check the orthogonality, factorization and the solution */
    info_solution = check_solution(N, NRHS, A1, LDA, B1, X, LDB, eps);

    if (info_solution == 0) {
        printf("***************************************************\n");
        printf(" ---- TESTING DSUNGESV.................... PASSED !\n");
        printf("***************************************************\n");
    }
    else {
        printf("************************************************\n");
        printf(" - TESTING DSUNGESV .. FAILED !\n");
        printf("************************************************\n");
    }

    free(A1); free(A2); free(B1); free(X); 

    PLASMA_Finalize();

    exit(0);
}

/*--------------------------------------------------------------
 * Check the solution
 */

int check_solution(int N, int NRHS, double *A1, int LDA, 
                   double *B1, double *B2, int LDB, double eps)
{
    int info_solution;
    double Rnorm, Anorm, Xnorm, Bnorm;
    double alpha, beta;

    double *Residual = (double *)malloc(N*NRHS*sizeof(double));
    double             *work     = (double *)            malloc(N     *sizeof(double));

    alpha = 1.0;
    beta  = -1.0;

    Anorm = lapack_dlange(lapack_inf_norm, N, N,    A1, LDA, work);
    Xnorm = lapack_dlange(lapack_inf_norm, N, NRHS, B2, LDB, work);
    Bnorm = lapack_dlange(lapack_inf_norm, N, NRHS, B1, LDB, work);

    cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, N, NRHS, N, 
                (alpha), A1, LDA, B2, LDB, (beta), B1, LDB);

    memset((void*)Residual, 0, N*NRHS*sizeof(double));

    cblas_dgemm(CblasColMajor, CblasTrans, CblasNoTrans, N, NRHS, N, 
                (alpha), A1, LDA, B1, LDB, (beta), Residual, N);

    Rnorm = lapack_dlange(lapack_inf_norm, N, NRHS, Residual, N, work);

    free(Residual);

    printf("============\n");
    printf("Checking the Residual of the solution \n");
    printf("-- ||Ax-B||_oo/((||A||_oo||x||_oo+||B||)_oo.N.eps) = %e \n", Rnorm/((Anorm*Xnorm+Bnorm)*N*eps));

    if (isnan(Rnorm / ((Anorm * Xnorm + Bnorm) * N * eps)) || isinf(Rnorm / ((Anorm * Xnorm + Bnorm) * N * eps)) || (Rnorm / ((Anorm * Xnorm + Bnorm) * N * eps) > 60.0) ) {
         printf("-- The solution is suspicious ! \n");
         info_solution = 1;
    }
    else {
         printf("-- The solution is CORRECT ! \n");
         info_solution= 0 ;
    }

    free(work);

    return info_solution;
}
