/**
 *
 * @file core_zgelqt.c
 *
 *  PLASMA core_blas kernel
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Hatem Ltaief
 * @author Mathieu Faverge
 * @date 2009-11-15
 *
 **/
#include <lapack.h>
#include "common.h"

/***************************************************************************//**
 *
 * @ingroup CORE_PLASMA_Complex64_t
 *
 *  CORE_zgelqt - computes a LQ factorization of a complex M-by-N tile A: A = L * Q.
 *
 *  The tile Q is represented as a product of elementary reflectors
 *
 *    Q = H(k)' . . . H(2)' H(1)', where k = min(M,N).
 *
 *  Each H(i) has the form
 *
 *    H(i) = I - tau * v * v'
 *
 *  where tau is a complex scalar, and v is a complex vector with
 *  v(1:i-1) = 0 and v(i) = 1; conjg(v(i+1:n)) is stored on exit in
 *  A(i,i+1:n), and tau in TAU(i).
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the tile A.  M >= 0.
 *
 * @param[in] N
 *         The number of columns of the tile A.  N >= 0.
 *
 * @param[in] IB
 *         The inner-blocking size.  IB >= 0.
 *
 * @param[in,out] A
 *         On entry, the M-by-N tile A.
 *         On exit, the elements on and below the diagonal of the array
 *         contain the M-by-min(M,N) lower trapezoidal tile L (L is
 *         lower triangular if M <= N); the elements above the diagonal,
 *         with the array TAU, represent the unitary tile Q as a
 *         product of elementary reflectors (see Further Details).
 *
 * @param[in] LDA
 *         The leading dimension of the array A.  LDA >= max(1,M).
 *
 * @param[out] T
 *         The IB-by-N triangular factor T of the block reflector.
 *         T is upper triangular by block (economic storage);
 *         The rest of the array is not referenced.
 *
 * @param[in] LDT
 *         The leading dimension of the array T. LDT >= IB.
 *
 * @param[out] TAU
 *         The scalar factors of the elementary reflectors (see Further
 *         Details).
 *
 * @param[in,out] WORK
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *          \retval <0 if -i, the i-th argument had an illegal value
 *
 ******************************************************************************/
int CORE_zgelqt(int M, int N, int IB,
                PLASMA_Complex64_t *A, int LDA,
                PLASMA_Complex64_t *T, int LDT,
                PLASMA_Complex64_t *TAU,
                PLASMA_Complex64_t *WORK)
{
    int i, k, sb;
    int tmp, iinfo;

    /* Check input arguments */
    if (M < 0) {
        plasma_error("CORE_zgelqt", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("CORE_zgelqt", "illegal value of N");
        return -2;
    }
    if (IB < 0) {
        plasma_error("CORE_zgelqt", "illegal value of IB");
        return -3;
    }
    if ((LDA < max(1,M)) && (M > 0)) {
        plasma_error("CORE_zgelqt", "illegal value of LDA");
        return -5;
    }
    if ((LDT < max(1,IB)) && (IB > 0)) {
        plasma_error("CORE_zgelqt", "illegal value of LDT");
        return -7;
    }

    /* Quick return */
    if ( (M == 0) || (N == 0) || (IB == 0))
        return PLASMA_SUCCESS;

    k = min(M, N);

    for(i=0; i<k; i+=IB) {
        sb = min(IB, k-i);

        tmp = N-i;
        lapack_zgelq2(sb, tmp, &A[LDA*i+i], LDA, &TAU[i], WORK, &iinfo);

        lapack_zlarft((enum lapack_direction_type)PlasmaForward,
                      (enum lapack_store_type)PlasmaRowwise,
                      tmp, sb,
                      &A[LDA*i+i], LDA, &TAU[i],
                      &T[LDT*i],   LDT);

        if ( M > ( i+IB ) ) {
            CORE_zunmlq(PlasmaRight, PlasmaConjTrans,
                        M-i-sb, N-i, sb, sb,
                        &A[LDA*i+i], LDA,
                        &T[LDT*i], LDT,
                        &A[LDA*i+(i+sb)], LDA,
                        WORK, M-i-sb);
        }
    }
    return PLASMA_SUCCESS;
}

/***************************************************************************//**
 *
 **/
void CORE_zgelqt_quark(Quark* quark)
{
    int M;
    int N;
    int IB;
    PLASMA_Complex64_t *A;
    int LDA;
    PLASMA_Complex64_t *T;
    int LDT;
    PLASMA_Complex64_t *TAU;
    PLASMA_Complex64_t *WORK;

    quark_unpack_args_9(quark, M, N, IB, A, LDA, T, LDT, TAU, WORK);
    CORE_zgelqt(M, N, IB, A, LDA, T, LDT, TAU, WORK);
}
