/**
 *
 * @file core_dssmqr.c
 *
 *  PLASMA core_blas kernel
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Hatem Ltaief
 * @author Mathieu Faverge
 * @date 2009-11-15
 *
 **/
#include "common.h"

/***************************************************************************//**
 *
 * @ingroup CORE_double
 *
 *  CORE_dssmqr overwrites the general complex M1-by-NN tile A1 and
 *  M2-by-NN tile A2 with
 *
 *                        SIDE = 'L'        SIDE = 'R'
 *    TRANS = 'N':         Q * | A1 |       | A1 | * Q
 *                             | A2 |       | A2 |
 *
 *    TRANS = 'C':      Q\*\*T * | A1 |       | A1 | * Q\*\*T
 *                             | A2 |       | A2 |
 *
 *  where Q is a complex unitary matrix defined as the product of k
 *  elementary reflectors
 *
 *    Q = H(1) H(2) . . . H(k)
 *
 *  as returned by CORE_ZTSQRT.
 *
 *  Only LEFT is supported !!!
 *
 *******************************************************************************
 *
 * @param[in] SIDE
 *         @arg PlasmaLeft  : apply Q or Q\*\*T from the Left;
 *         @arg PlasmaRight : apply Q or Q\*\*T from the Right.
 *
 * @param[in] TRANS
 *         @arg PlasmaNoTrans   :  No transpose, apply Q;
 *         @arg PlasmaTrans :  ConjTranspose, apply Q\*\*T.
 *
 * @param[in] M1
 *         The number of columns of the tile A1. M1 >= 0.
 *
 * @param[in] M2
 *         The number of columns of the tile A2. M2 >= 0.
 *
 * @param[in] NN
 *         The number of rows of the tiles A1 and A2. NN >= 0.
 *
 * @param[in] IB
 *         The inner-blocking size.  IB >= 0.
 *
 * @param[in] K
 *         The number of elementary reflectors whose product defines
 *         the matrix Q.
 *
 * @param[in,out] A1
 *         On entry, the M1-by-NN tile A1.
 *         On exit, A1 is overwritten by the application of Q.
 *
 * @param[in] LDA1
 *         The leading dimension of the array A1. LDA1 >= max(1,M1).
 *
 * @param[in,out] A2
 *         On entry, the M2-by-NN tile A2.
 *         On exit, A2 is overwritten by the application of Q.
 *
 * @param[in] LDA2
 *         The leading dimension of the tile A2. LDA2 >= max(1,M2).
 *
 * @param[in] V
 *         The i-th row must contain the vector which defines the
 *         elementary reflector H(i), for i = 1,2,...,k, as returned by
 *         CORE_ZTSQRT in the first k rows of its array argument V.
 *
 * @param[in] LDV
 *         The leading dimension of the array V. LDV >= max(1,K).
 *
 * @param[out] T
 *         The IB-by-NN triangular factor T of the block reflector.
 *         T is upper triangular by block (economic storage);
 *         The rest of the array is not referenced.
 *
 * @param[in] LDT
 *         The leading dimension of the array T. LDT >= IB.
 *
 * @param[in,out] WORK
 *
 * @param[in] LDWORK
 *         The dimension of the array WORK.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *          \retval <0 if -i, the i-th argument had an illegal value
 *
 ******************************************************************************/
int CORE_dssmqr(int side, int trans,
                int M1, int M2, int NN, int IB, int K,
                double *A1,   int LDA1,
                double *A2,   int LDA2,
                double *V,    int LDV,
                double *T,    int LDT,
                double *WORK, int LDWORK)
{
    int i,  i1, i3;
    int nq; /*, nw;*/
    int kb;
    int ic = 0;
    int jc = 0;
    int mi = M1;
    int ni = NN;

    /* Check input arguments */
    if ((side != PlasmaLeft) && (side != PlasmaRight)) {
        plasma_error("CORE_dssmqr", "illegal value of side");
        return -1;
    }
    /* NQ is the order of Q and NW is the minimum dimension of WORK */
    if (side == PlasmaLeft) {
        nq = M1;
        /* nw = NN; */
    }
    else {
        nq = NN;
        /* nw = M1; */
    }

    if ((trans != PlasmaNoTrans) && (trans != PlasmaTrans)) {
        plasma_error("CORE_dssmqr", "illegal value of trans");
        return -2;
    }
    if (M1 < 0) {
        plasma_error("CORE_dssmqr", "illegal value of M1");
        return -3;
    }
    if (M2 < 0) {
        plasma_error("CORE_dssmqr", "illegal value of M2");
        return -4;
    }
    if (NN < 0) {
        plasma_error("CORE_dssmqr", "illegal value of NN");
        return -5;
    }
    if (IB < 0) {
        plasma_error("CORE_dssmqr", "illegal value of IB");
        return -6;
    }
    if ((K < 0) || (K > nq)) {
        plasma_error("CORE_dssmqr", "illegal value of K");
        return -7;
    }
    if ((LDA1 < max(1,nq)) && (nq > 0)){
        plasma_error("CORE_dssmqr", "illegal value of LDA1");
        return -9;
    }

    /* Quick return */
    if ( (M1 == 0) || (M2 == 0) || (NN == 0) || (K == 0))
        return PLASMA_SUCCESS;

    if (((side == PlasmaLeft) && (trans != PlasmaNoTrans))
        || ((side == PlasmaRight) && (trans == PlasmaNoTrans))) {
        i1 = 0;
        /* i2 = K; */
        i3 = IB;
    }
    else {
        i1 = ( ( K-1 ) / IB )*IB;
        /* i2 = 0; */
        i3 = -IB;
    }

    for(i=i1; (i>-1) && (i<K); i+=i3 ) {
        kb = min(IB, K-i);

        if (side == PlasmaLeft) {
            /*
             * H or H' is applied to C(i:m,1:n)
             */
            mi = M1 - i;
            ic = i;
        }
        else {
            /*
             * H or H' is applied to C(1:m,i:n)
             */
            ni = NN - i;
            jc = i;
        }
        /*
         * Apply H or H'
         */
        CORE_dssrfb(side, trans, PlasmaForward, PlasmaColumnwise,
                    mi, M2, ni, kb,
                    &A1[LDA1*jc+ic], LDA1,
                    A2,              LDA2,
                    &V[LDV*i],       LDV,
                    &T[LDT*i],       LDT,
                    WORK,            LDWORK);
    }
    return PLASMA_SUCCESS;
}

/***************************************************************************//**
 *
 **/
void CORE_dssmqr_quark(Quark* quark)
{
    int side;
    int trans;
    int M1;
    int M2;
    int NN;
    int IB;
    int K;
    double *A1;
    int LDA1;
    double *A2;
    int LDA2;
    double *V;
    int LDV;
    double *T;
    int LDT;
    double *WORK;
    int LDWORK;

    quark_unpack_args_17(quark, side, trans, M1, M2, NN, IB, K, A1, LDA1, A2, LDA2, V, LDV, T, LDT, WORK, LDWORK);
    CORE_dssmqr(side, trans, M1, M2, NN, IB, K, A1, LDA1, A2, LDA2, V, LDV, T, LDT, WORK, LDWORK);
}
