/**
 *
 * @file core_dormqr.c
 *
 *  PLASMA core_blas kernel
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Hatem Ltaief
 * @author Mathieu Faverge
 * @date 2009-11-15
 *
 **/
#include <lapack.h>
#include "common.h"

/***************************************************************************//**
 *
 * @ingroup CORE_double
 *
 *  CORE_dormqr overwrites the general complex M-by-N tile C with
 *
 *                    SIDE = 'L'     SIDE = 'R'
 *    TRANS = 'N':      Q * C          C * Q
 *    TRANS = 'C':      Q\*\*T * C       C * Q\*\*T
 *
 *  where Q is a complex unitary matrix defined as the product of k
 *  elementary reflectors
 *
 *    Q = H(1) H(2) . . . H(k)
 *
 *  as returned by CORE_dgeqrt. Q is of order M if SIDE = 'L' and of order N
 *  if SIDE = 'R'.
 *
 *******************************************************************************
 *
 * @param[in] SIDE
 *         @arg PlasmaLeft  : apply Q or Q\*\*T from the Left;
 *         @arg PlasmaRight : apply Q or Q\*\*T from the Right.
 *
 * @param[in] TRANS
 *         @arg PlasmaNoTrans   :  No transpose, apply Q;
 *         @arg PlasmaTrans :  Transpose, apply Q\*\*T.
 *
 * @param[in] M
 *         The number of rows of the tile C.  M >= 0.
 *
 * @param[in] N
 *         The number of columns of the tile C.  N >= 0.
 *
 * @param[in] IB
 *         The inner-blocking size.  IB >= 0.
 *
 * @param[in] K
 *         The number of elementary reflectors whose product defines
 *         the matrix Q.
 *         If SIDE = PlasmaLeft,  M >= K >= 0;
 *         if SIDE = PlasmaRight, N >= K >= 0.
 *
 * @param[in] A
 *         The i-th column must contain the vector which defines the
 *         elementary reflector H(i), for i = 1,2,...,k, as returned by
 *         CORE_dgeqrt in the first k columns of its array argument A.
 *
 * @param[in] LDA
 *         The leading dimension of the array A.
 *         If SIDE = PlasmaLeft,  LDA >= max(1,M);
 *         if SIDE = PlasmaRight, LDA >= max(1,N).
 *
 * @param[out] T
 *         The IB-by-K triangular factor T of the block reflector.
 *         T is upper triangular by block (economic storage);
 *         The rest of the array is not referenced.
 *
 * @param[in] LDT
 *         The leading dimension of the array T. LDT >= IB.
 *
 * @param[in,out] C
 *         On entry, the M-by-N tile C.
 *         On exit, C is overwritten by Q*C or Q**T*C or C*Q**T or C*Q.
 *
 * @param[in] LDC
 *         The leading dimension of the array C. LDC >= max(1,M).
 *
 * @param[in,out] WORK
 *         On exit, if INFO = 0, WORK(1) returns the optimal LDWORK.
 *
 * @param[in] LDWORK
 *         The dimension of the array WORK.
 *         If SIDE = PlasmaLeft,  LDWORK >= max(1,N);
 *         if SIDE = PlasmaRight, LDWORK >= max(1,M).
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *          \retval <0 if -i, the i-th argument had an illegal value
 *
 ******************************************************************************/
int CORE_dormqr(int side, int trans,
                int M, int N, int IB, int K,
                double *A,    int LDA,
                double *T,    int LDT,
                double *C,    int LDC,
                double *WORK, int LDWORK)
{
    int i, kb;
    int i1, i3;
    int nq, nw;
    int ic = 0;
    int jc = 0;
    int ni = N;
    int mi = M;

    /* Check input arguments */
    if ((side != PlasmaLeft) && (side != PlasmaRight)) {
        plasma_error("CORE_dormqr", "illegal value of side");
        return -1;
    }
    /*
     * NQ is the order of Q and NW is the minimum dimension of WORK
     */
    if (side == PlasmaLeft) {
        nq = M;
        nw = N;
    }
    else {
        nq = N;
        nw = M;
    }
    if ((trans != PlasmaNoTrans) && (trans != PlasmaTrans)) {
        plasma_error("CORE_dormqr", "illegal value of trans");
        return -2;
    }
    if (M < 0) {
        plasma_error("CORE_dormqr", "illegal value of M");
        return -3;
    }
    if (N < 0) {
        plasma_error("CORE_dormqr", "illegal value of N");
        return -4;
    }
    if ((IB < 0) || ( (IB == 0) && ((M > 0) && (N > 0)) )) {
        plasma_error("CORE_dormqr", "illegal value of IB");
        return -5;
    }
    if ((K < 0) || (K > nq)) {
        plasma_error("CORE_dormqr", "illegal value of K");
        return -6;
    }
    if ((LDA < max(1,nq)) && (nq > 0)) {
        plasma_error("CORE_dormqr", "illegal value of LDA");
        return -8;
    }
    if ((LDC < max(1,M)) && (M > 0)) {
        plasma_error("CORE_dormqr", "illegal value of LDC");
        return -12;
    }
    if ((LDWORK < max(1,nw)) && (nw > 0)) {
        plasma_error("CORE_dormqr", "illegal value of LDWORK");
        return -14;
    }

    /* Quick return */
    if ( (M == 0) || (N == 0) || (K == 0))
        return PLASMA_SUCCESS;

    if (((side == PlasmaLeft) && (trans != PlasmaNoTrans))
        || ((side == PlasmaRight) && (trans == PlasmaNoTrans))) {
        i1 = 0;
        /* i2 = K; */
        i3 = IB;
    }
    else {
        i1 = ( ( K-1 ) / IB )*IB;
        /* i2 = 0; */
        i3 = -IB;
    }

    for(i=i1; (i>-1) && (i<K); i+=i3 ) {
        kb = min(IB, K-i);

        if (side == PlasmaLeft) {
            /*
             * H or H' is applied to C(i:m,1:n)
             */
            mi = M - i;
            ic = i;
        }
        else {
            /*
             * H or H' is applied to C(1:m,i:n)
             */
            ni = N - i;
            jc = i;
        }
        /*
         * Apply H or H'
         */
        lapack_dlarfb((enum lapack_side_type)side,
                      (enum lapack_trans_type)trans,
                      (enum lapack_direction_type)PlasmaForward,
                      (enum lapack_store_type)PlasmaColumnwise,
                      mi, ni, kb,
                      &A[LDA*i+i],   LDA,
                      &T[LDT*i],     LDT,
                      &C[LDC*jc+ic], LDC,
                      WORK, LDWORK);
    }
    return PLASMA_SUCCESS;
}

/***************************************************************************//**
 *
 **/
void CORE_dormqr_quark(Quark* quark)
{
    int side;
    int trans;
    int M;
    int N;
    int IB;
    int K;
    double *A;
    int LDA;
    double *T;
    int LDT;
    double *C;
    int LDC;
    double *WORK;
    int LDWORK;

    quark_unpack_args_14(quark, side, trans, M, N, IB, K, A, LDA, T, LDT, C, LDC, WORK, LDWORK);
    CORE_dormqr(side, trans, M, N, IB, K, A, LDA, T, LDT, C, LDC, WORK, LDWORK);
}
