/**
 *
 * @file workspace_c.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Jakub Kurzak
 * @date 2009-11-15
 *
 **/
#include "common.h"
#include "workspace.h"

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgels - Allocates workspace for PLASMA_cgels or PLASMA_cgels_Tile routine.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] T
 *          On exit, workspace handle for storage of the extra T factors required by the tile QR
 *          or the tile LQ factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgels(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgels_Tile - Allocates tile workspace for PLASMA_cgels_Tile routine.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] descT
 *          On exit, PLASMA_desc on workspace handle for storage of the extra T factors required by the tile QR
 *          or the tile LQ factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgels_Tile(int M, int N, PLASMA_desc **descT) {
    return plasma_alloc_ibnb_tile(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, descT); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgeqrf - Allocates workspace for PLASMA_cgeqrf or PLASMA_cgeqrf_Tile routine.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] T
 *          On exit, workspace handle for storage of the extra T factors required by the tile QR
 *          factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgeqrf(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgeqrf_Tile - Allocates tile workspace for PLASMA_cgels_Tile routine.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] descT
 *          On exit, PLASMA_desc on workspace handle for storage of the extra T factors required by the tile QR
 *          or the tile LQ factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgeqrf_Tile(int M, int N, PLASMA_desc **descT) {
    return plasma_alloc_ibnb_tile(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, descT); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgelqf - Allocates workspace for PLASMA_cgelqf or PLASMA_cgelqf_Tile routines.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] T
 *          On exit, workspace handle for storage of the extra T factors required by the tile LQ
 *          factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgelqf(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgelqf_Tile - Allocates tile workspace for PLASMA_cgels_Tile routine.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] descT
 *          On exit, PLASMA_desc on workspace handle for storage of the extra T factors required by the tile QR
 *          or the tile LQ factorization.
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgelqf_Tile(int M, int N, PLASMA_desc **descT) {
    return plasma_alloc_ibnb_tile(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, descT); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgesv - Allocates workspace for PLASMA_cgesv or PLASMA_cgesv_Tile routines.
 *
 *******************************************************************************
 *
 * @param[in] N
 *          The number of linear equations, i.e., the order of the matrix A. N >= 0.
 *
 * @param[out] L
 *          On exit, workspace handle for storage of the extra L factors required by the tile LU
 *          factorization.
 *
 * @param[out] IPIV
 *          On exit, workspace handle for storage of pivot indexes required by the tile LU
 *          factorization (not equivalent to LAPACK).
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgesv(int N, PLASMA_Complex32_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(N, N, PLASMA_FUNC_CGESV, PlasmaComplexFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_CGESV, (void**)IPIV); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgesv_Tile - Allocates workspace for PLASMA_cgesv_Tile routines.
 *
 *******************************************************************************
 *
 * @param[in] N
 *          The number of linear equations, i.e., the order of the matrix A. N >= 0.
 *
 * @param[out] descL
 *          On exit, PLASMA descriptor on workspace handle for storage of the extra
 *          L factors required by the tile LU factorization.
 *
 * @param[out] IPIV
 *          On exit, workspace handle for storage of pivot indexes required by the tile LU
 *          factorization (not equivalent to LAPACK).
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgesv_Tile(int N, PLASMA_desc **descL, int **IPIV)
{
    int status = plasma_alloc_ibnb_tile(N, N, PLASMA_FUNC_SGESV, PlasmaComplexFloat, descL);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_CGESV, (void **)IPIV);
}

/***************************************************************************//**
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgetrf - Allocates workspace for PLASMA_cgetrf or PLASMA_cgetrf_Tile routines.
 *
 *******************************************************************************
 *
 * @param[in] M
 *          The number of rows of the matrix A. M >= 0.
 *
 * @param[in] N
 *          The number of columns of the matrix A.  N >= 0.
 *
 * @param[out] L
 *          On exit, workspace handle for storage of the extra L factors required by the tile LU
 *          factorization.
 *
 * @param[out] IPIV
 *          On exit, workspace handle for storage of pivot indexes required by the tile LU
 *          factorization (not equivalent to LAPACK).
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgetrf(int M, int N, PLASMA_Complex32_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGESV, PlasmaComplexFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(M, N, PLASMA_FUNC_CGESV, (void**)IPIV); }

/***************************************************************************//**
 *
 * @ingroup Auxiliary
 *
 *  PLASMA_Alloc_Workspace_cgetrf_Tile - Allocates workspace for PLASMA_cgesv_Tile routines.
 *
 *******************************************************************************
 *
 * @param[in] N
 *          The number of linear equations, i.e., the order of the matrix A. N >= 0.
 *
 * @param[out] descL
 *          On exit, PLASMA descriptor on workspace handle for storage of the extra
 *          L factors required by the tile LU factorization.
 *
 * @param[out] IPIV
 *          On exit, workspace handle for storage of pivot indexes required by the tile LU
 *          factorization (not equivalent to LAPACK).
 *
 *******************************************************************************
 *
 * @return
 *          \retval PLASMA_SUCCESS successful exit
 *
 ******************************************************************************/
int PLASMA_Alloc_Workspace_cgetrf_Tile(int N, PLASMA_desc **descL, int **IPIV)
{
    int status = plasma_alloc_ibnb_tile(N, N, PLASMA_FUNC_SGESV, PlasmaComplexFloat, descL);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_CGESV, (void **)IPIV);
}

