/**
 *
 * @file common.h
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Jakub Kurzak
 * @date 2009-11-15
 *
 **/

/***************************************************************************//**
 *  PLASMA facilities of interest to both PLASMA core developer
 *  and also of interest to PLASMA community contributor.
 **/
#ifndef _PLASMA_COMMON_H_
#define _PLASMA_COMMON_H_

#include "global.h"
#include "context.h"
#include "control.h"
#include "core_blas.h"
#include "allocate.h"
#include "auxiliary.h"
#include "descriptor.h"
#include "tile.h"
#include "async.h"

/***************************************************************************//**
 *  Global shortcuts
 **/
#define PLASMA_RANK       plasma_rank(plasma)
#define PLASMA_SIZE       plasma->world_size
#define PLASMA_NB         plasma->nb
#define PLASMA_IB         plasma->ib
#define PLASMA_NBNBSIZE   plasma->nbnbsize
#define PLASMA_IBNBSIZE   plasma->ibnbsize
#define PLASMA_INFO       plasma->info
#define PLASMA_SCHEDULING plasma->scheduling

/***************************************************************************//**
 *  Global utilities
 **/
#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif
#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif
#ifndef roundup
#define roundup(a, b) (b <= 0) ? (a) : (((a) + (b)-1) & ~((b)-1))
#endif

/***************************************************************************//**
 *  Static scheduler
 **/
#define ss_init(m, n, init_val) \
{ \
    if (PLASMA_RANK == 0) { \
        plasma->ss_progress = (volatile int *)plasma_shared_alloc(plasma, (m)*(n), PlasmaInteger); \
        plasma_memset_int((int*)plasma->ss_progress, (m)*(n), (init_val)); \
    } \
    plasma->ss_ld = (m); \
    plasma->ss_abort = 0; \
    plasma_barrier(plasma); \
}

#define ss_abort()   plasma->ss_abort = 1;
#define ss_aborted() plasma->ss_abort

#define ss_cond_set(m, n, val) \
{ \
    plasma->ss_progress[(m)+plasma->ss_ld*(n)] = (val); \
}

#define ss_cond_wait(m, n, val) \
{ \
    while (!plasma->ss_abort && \
            plasma->ss_progress[(m)+plasma->ss_ld*(n)] != (val)) \
        plasma_yield(); \
    if (plasma->ss_abort) \
        break; \
}

#define ss_finalize() \
{ \
    plasma_barrier(plasma); \
    if (PLASMA_RANK == 0) \
        plasma_shared_free(plasma, (void*)plasma->ss_progress); \
}

/***************************************************************************//**
 *  Global array of LAPACK constants
 **/
extern char *plasma_lapack_constants[];

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************//**
 *  Parallel numerical functions prototypes - static scheduling
 **/
void plasma_psgelqf(plasma_context_t *plasma);
void plasma_psgeqrf(plasma_context_t *plasma);
void plasma_psgetrf(plasma_context_t *plasma);
void plasma_psorglq(plasma_context_t *plasma);
void plasma_psorgqr(plasma_context_t *plasma);
void plasma_psormlq(plasma_context_t *plasma);
void plasma_psormqr(plasma_context_t *plasma);
void plasma_pspotrf(plasma_context_t *plasma);
void plasma_pstrsm(plasma_context_t *plasma);
void plasma_pstrsmpl(plasma_context_t *plasma);
void plasma_psgemm(plasma_context_t *plasma);
void plasma_pssymm(plasma_context_t *plasma);
void plasma_pssyrk(plasma_context_t *plasma);

void plasma_pdgelqf(plasma_context_t *plasma);
void plasma_pdgeqrf(plasma_context_t *plasma);
void plasma_pdgetrf(plasma_context_t *plasma);
void plasma_pdorglq(plasma_context_t *plasma);
void plasma_pdorgqr(plasma_context_t *plasma);
void plasma_pdormlq(plasma_context_t *plasma);
void plasma_pdormqr(plasma_context_t *plasma);
void plasma_pdpotrf(plasma_context_t *plasma);
void plasma_pdtrsm(plasma_context_t *plasma);
void plasma_pdtrsmpl(plasma_context_t *plasma);
void plasma_pdgemm(plasma_context_t *plasma);
void plasma_pdsymm(plasma_context_t *plasma);
void plasma_pdsyrk(plasma_context_t *plasma);

void plasma_pcgelqf(plasma_context_t *plasma);
void plasma_pcgeqrf(plasma_context_t *plasma);
void plasma_pcgetrf(plasma_context_t *plasma);
void plasma_pcunglq(plasma_context_t *plasma);
void plasma_pcungqr(plasma_context_t *plasma);
void plasma_pcunmlq(plasma_context_t *plasma);
void plasma_pcunmqr(plasma_context_t *plasma);
void plasma_pcpotrf(plasma_context_t *plasma);
void plasma_pctrsm(plasma_context_t *plasma);
void plasma_pctrsmpl(plasma_context_t *plasma);
void plasma_pcgemm(plasma_context_t *plasma);
void plasma_pcsymm(plasma_context_t *plasma);
void plasma_pchemm(plasma_context_t *plasma);
void plasma_pcsyrk(plasma_context_t *plasma);
void plasma_pcherk(plasma_context_t *plasma);

void plasma_pzgelqf(plasma_context_t *plasma);
void plasma_pzgeqrf(plasma_context_t *plasma);
void plasma_pzgetrf(plasma_context_t *plasma);
void plasma_pzunglq(plasma_context_t *plasma);
void plasma_pzungqr(plasma_context_t *plasma);
void plasma_pzunmlq(plasma_context_t *plasma);
void plasma_pzunmqr(plasma_context_t *plasma);
void plasma_pzpotrf(plasma_context_t *plasma);
void plasma_pztrsm(plasma_context_t *plasma);
void plasma_pztrsmpl(plasma_context_t *plasma);
void plasma_pzgemm(plasma_context_t *plasma);
void plasma_pzsymm(plasma_context_t *plasma);
void plasma_pzhemm(plasma_context_t *plasma);
void plasma_pzsyrk(plasma_context_t *plasma);
void plasma_pzherk(plasma_context_t *plasma);

/***************************************************************************//**
 *  Parallel numerical functions prototypes - dynamic scheduling
 **/
void plasma_psgetrf_quark(PLASMA_desc A, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pstrsmpl_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pstrsm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, float alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pspotrf_quark(PLASMA_enum uplo, PLASMA_desc A, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psgeqrf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psgelqf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psorgqr_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psorglq_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psormqr_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psormlq_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_psgemm_quark(PLASMA_enum transA, PLASMA_enum transB, float alpha, PLASMA_desc A, PLASMA_desc B, float beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pssymm_quark(PLASMA_enum side, PLASMA_enum uplo, float alpha, PLASMA_desc A, PLASMA_desc B, float beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pssyrk_quark(PLASMA_enum uplo, PLASMA_enum trans, float alpha, PLASMA_desc A, float beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);

void plasma_pdgetrf_quark(PLASMA_desc A, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdtrsmpl_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdtrsm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, double alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdpotrf_quark(PLASMA_enum uplo, PLASMA_desc A, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdgeqrf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdgelqf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdorgqr_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdorglq_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdormqr_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdormlq_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdgemm_quark(PLASMA_enum transA, PLASMA_enum transB, double alpha, PLASMA_desc A, PLASMA_desc B, double beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdsymm_quark(PLASMA_enum side, PLASMA_enum uplo, double alpha, PLASMA_desc A, PLASMA_desc B, double beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pdsyrk_quark(PLASMA_enum uplo, PLASMA_enum trans, double alpha, PLASMA_desc A, double beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);

void plasma_pcgetrf_quark(PLASMA_desc A, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pctrsmpl_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pctrsm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcpotrf_quark(PLASMA_enum uplo, PLASMA_desc A, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcgeqrf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcgelqf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcungqr_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcunglq_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcunmqr_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcunmlq_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcgemm_quark(PLASMA_enum transA, PLASMA_enum transB, PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex32_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcsymm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex32_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pchemm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex32_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcsyrk_quark(PLASMA_enum uplo, PLASMA_enum trans, PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_Complex32_t beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pcherk_quark(PLASMA_enum uplo, PLASMA_enum trans, float alpha, PLASMA_desc A, float beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);

void plasma_pzgetrf_quark(PLASMA_desc A, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pztrsmpl_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc L, int *IPIV, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pztrsm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, PLASMA_Complex64_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzpotrf_quark(PLASMA_enum uplo, PLASMA_desc A, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzgeqrf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzgelqf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzungqr_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzunglq_quark(PLASMA_desc A, PLASMA_desc Q, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzunmqr_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzunmlq_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzgemm_quark(PLASMA_enum transA, PLASMA_enum transB, PLASMA_Complex64_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex64_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzsymm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_Complex64_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex64_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzhemm_quark(PLASMA_enum side, PLASMA_enum uplo, PLASMA_Complex64_t alpha, PLASMA_desc A, PLASMA_desc B, PLASMA_Complex64_t beta, PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzsyrk_quark(PLASMA_enum uplo, PLASMA_enum trans, PLASMA_Complex64_t alpha, PLASMA_desc A, PLASMA_Complex64_t beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);
void plasma_pzherk_quark(PLASMA_enum uplo, PLASMA_enum trans, double alpha, PLASMA_desc A, double beta,  PLASMA_desc C, PLASMA_sequence *sequence, PLASMA_request *request);

/***************************************************************************//**
 *  Parallel auxiliary functions prototypes
 **/
void plasma_pslange(plasma_context_t *plasma);
void plasma_pslansy(plasma_context_t *plasma);
void plasma_pslacpy(plasma_context_t *plasma);
void plasma_psaxpy(plasma_context_t *plasma);

void plasma_pdlange(plasma_context_t *plasma);
void plasma_pdlansy(plasma_context_t *plasma);
void plasma_pdlacpy(plasma_context_t *plasma);
void plasma_pdaxpy(plasma_context_t *plasma);

void plasma_pclange(plasma_context_t *plasma);
void plasma_pclanhe(plasma_context_t *plasma);
void plasma_pclacpy(plasma_context_t *plasma);
void plasma_pcaxpy(plasma_context_t *plasma);

void plasma_pzlange(plasma_context_t *plasma);
void plasma_pzlanhe(plasma_context_t *plasma);
void plasma_pzlacpy(plasma_context_t *plasma);
void plasma_pzaxpy(plasma_context_t *plasma);

void plasma_pdlag2s(plasma_context_t *plasma);
void plasma_pzlag2c(plasma_context_t *plasma);
void plasma_pslag2d(plasma_context_t *plasma);
void plasma_pclag2z(plasma_context_t *plasma);


#ifdef __cplusplus
}
#endif

#endif
