/**
 *
 * @file pztrsmpl.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Jakub Kurzak
 * @author Hatem Ltaief
 * @date 2009-11-15
 *
 **/
#include "common.h"

#define A(m,n)    BLKADDR(A, PLASMA_Complex64_t, m, n)
#define B(m,n)    BLKADDR(B, PLASMA_Complex64_t, m, n)
#define L(m,n)    BLKADDR(L, PLASMA_Complex64_t, m, n)
#define IPIV(m,n) &(IPIV[(int64_t)A.nb*((int64_t)(m)+(int64_t)A.lmt*(int64_t)(n))])
/***************************************************************************//**
 *  Parallel application of Q using tile V (QR factorization) - static scheduling
 **/
void plasma_pztrsmpl(plasma_context_t *plasma)
{
    PLASMA_desc A;
    PLASMA_desc B;
    PLASMA_desc L;
    int *IPIV;
    PLASMA_sequence *sequence;
    PLASMA_request *request;

    int k, m, n;
    int next_k;
    int next_m;
    int next_n;

    plasma_unpack_args_6(A, B, L, IPIV, sequence, request);
    if (sequence->status != PLASMA_SUCCESS)
        return;
    ss_init(B.mt, B.nt, -1);

    k = 0;
    n = PLASMA_RANK;
    while (n >= B.nt) {
        k++;
        n = n-B.nt;
    }
    m = k;

    while (k < min(A.mt, A.nt) && n < B.nt) {
        next_n = n;
        next_m = m;
        next_k = k;

        next_m++;
        if (next_m == A.mt) {
            next_n += PLASMA_SIZE;
            while (next_n >= B.nt && next_k < min(A.mt, A.nt)) {
                next_k++;
                next_n = next_n-B.nt;
            }
            next_m = next_k;
        }

        if (m == k) {
            ss_cond_wait(k, n, k-1);
            CORE_zgessm(
                k == A.mt-1 ? A.m-k*A.nb : A.nb,
                n == B.nt-1 ? B.n-n*B.nb : B.nb,
                k == A.nt-1 ? min(A.m-k*A.nb, A.n-k*A.nb) : A.nb,
                L.mb,
                IPIV(k, k),
                A(k, k), A.nb,
                B(k, n), B.nb);
            ss_cond_set(k, n, k);
        }
        else {
            ss_cond_wait(m, n, k-1);
            CORE_zssssm(
                A.nb,
                m == A.mt-1 ? A.m-m*A.nb : A.nb,
                n == B.nt-1 ? B.n-n*A.nb : B.nb,
                L.mb,
                k == A.nt-1 ? A.n-k*A.nb : A.nb,
                B(k, n), B.nb,
                B(m, n), B.nb,
                L(m, k), L.mb,
                A(m, k), A.nb,
                IPIV(m, k));
            ss_cond_set(m, n, k);
        }
        n = next_n;
        m = next_m;
        k = next_k;
    }
    ss_finalize();
}

/***************************************************************************//**
 *  Parallel application of Q using tile V (QR factorization) - dynamic scheduling
 **/
void plasma_pztrsmpl_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc L, int *IPIV,
                            PLASMA_sequence *sequence, PLASMA_request *request)
{
    int k, m, n;
    plasma_context_t *plasma;
    int temp1, temp2, temp3;
    Quark_Task_Flags task_flags = Quark_Task_Flags_Initializer;

    plasma = plasma_context_self();
    if (sequence->status != PLASMA_SUCCESS)
        return;
    QUARK_Task_Flag_Set(&task_flags, TASK_SEQUENCE, (intptr_t)sequence->quark_sequence);

    for (k = 0; k < min(A.mt, A.nt); k++)
    {
        for (n = 0; n < B.nt; n++)
        {
            temp1 = (k == A.mt-1) ? A.m - k * A.mb : A.mb;
            temp2 = (n == B.nt-1) ? B.n - n * B.nb : B.nb;
            temp3 = (k == A.nt-1) ? min(A.m-k*A.nb, A.n-k*A.nb) : A.nb;

            QUARK_Insert_Task(plasma->quark, CORE_zgessm_quark, &task_flags,
                sizeof(int),                          &temp1,     VALUE,
                sizeof(int),                          &temp2,     VALUE,
                sizeof(int),                          &temp3,     VALUE,
                sizeof(int),                          &L.mb,      VALUE,
                sizeof(int)*A.mb,                     IPIV(k, k),     INPUT,
                sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, k),        INPUT,
                sizeof(int),                          &A.nb,      VALUE,
                sizeof(PLASMA_Complex64_t)*B.mb*B.nb, B(k, n),        INOUT | LOCALITY,
                sizeof(int),                          &B.nb,      VALUE,
                0);
        }
        for (m = k+1; m < A.mt; m++)
        {
            for (n = 0; n < B.nt; n++)
            {
                temp1 = (m == A.mt-1) ? A.m - m * A.mb : A.mb;
                temp2 = (n == B.nt-1) ? B.n - n * B.nb : B.nb;
                temp3 = (k == A.nt-1) ? A.n - k * A.nb : A.nb;

                QUARK_Insert_Task(plasma->quark, CORE_zssssm_quark, &task_flags,
                    sizeof(int),                          &A.nb,      VALUE,
                    sizeof(int),                          &temp1,     VALUE,
                    sizeof(int),                          &temp2,     VALUE,
                    sizeof(int),                          &L.mb,      VALUE,
                    sizeof(int),                          &temp3,     VALUE,
                    sizeof(PLASMA_Complex64_t)*B.mb*B.nb, B(k, n),        INOUT | LOCALITY,
                    sizeof(int),                          &B.nb,      VALUE,
                    sizeof(PLASMA_Complex64_t)*B.mb*B.nb, B(m, n),        INOUT,
                    sizeof(int),                          &B.nb,      VALUE,
                    sizeof(PLASMA_Complex64_t)*L.mb*L.nb, L(m, k),        INPUT,
                    sizeof(int),                          &L.mb,      VALUE,
                    sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(m, k),        INPUT,
                    sizeof(int),                          &A.nb,      VALUE,
                    sizeof(int)*A.mb,                     IPIV(m, k),     INPUT,
                    0);
            }
        }
    }
}
