/**
 *
 * @file pzgelqf.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Jakub Kurzak
 * @author Hatem Ltaief
 * @date 2009-11-15
 *
 **/
#include "common.h"

#define A(m,n) BLKADDR(A, PLASMA_Complex64_t, m, n)
#define T(m,n) BLKADDR(T, PLASMA_Complex64_t, m, n)
/***************************************************************************//**
 *  Parallel tile LQ factorization - static scheduling
 **/
void plasma_pzgelqf(plasma_context_t *plasma)
{
    PLASMA_desc A;
    PLASMA_desc T;
    PLASMA_sequence *sequence;
    PLASMA_request *request;

    int k, m, n;
    int next_k;
    int next_m;
    int next_n;
    PLASMA_Complex64_t *work, *tau;

    plasma_unpack_args_4(A, T, sequence, request);
    if (sequence->status != PLASMA_SUCCESS)
        return;
    work = (PLASMA_Complex64_t*)plasma_private_alloc(plasma, T.mb*T.nb, T.dtyp);
    tau  = (PLASMA_Complex64_t*)plasma_private_alloc(plasma, A.nb, A.dtyp);
    ss_init(A.mt, A.nt, -1);

    k = 0;
    m = PLASMA_RANK;
    while (m >= A.mt) {
        k++;
        m = m-A.mt+k;
    }
    n = k;

    while (k < min(A.mt, A.nt) && m < A.mt) {
        next_m = m;
        next_n = n;
        next_k = k;

        next_n++;
        if (next_n == A.nt) {
            next_m += PLASMA_SIZE;
            while (next_m >= A.mt && next_k < min(A.nt, A.mt)) {
                next_k++;
                next_m = next_m-A.mt+next_k;
            }
            next_n = next_k;
        }

        if (m == k) {
            if (n == k) {
                ss_cond_wait(k, k, k-1);
                CORE_zgelqt(
                    k == A.mt-1 ? A.m-k*A.nb : A.nb,
                    k == A.nt-1 ? A.n-k*A.nb : A.nb,
                    T.mb,
                    A(k, k), A.nb,
                    T(k, k), T.mb,
                    tau, work);
                ss_cond_set(k, k, k);
            }
            else {
                ss_cond_wait(k, n, k-1);
                CORE_ztslqt(
                    k == A.mt-1 ? A.m-k*A.nb : A.nb,
                    n == A.nt-1 ? A.n-n*A.nb : A.nb,
                    T.mb,
                    A(k, k), A.nb,
                    A(k, n), A.nb,
                    T(k, n), T.mb,
                    tau, work);
                ss_cond_set(k, n, k);
            }
        }
        else {
            if (n == k) {
                ss_cond_wait(k, k, k);
                ss_cond_wait(m, k, k-1);
                CORE_zunmlq(
                    PlasmaRight, PlasmaConjTrans,
                    m == A.mt-1 ? A.m-m*A.nb : A.nb,
                    k == A.nt-1 ? A.n-k*A.nb : A.nb,
                    T.mb,
                    k == A.nt-1 ? A.n-k*A.nb : A.nb,
                    A(k, k), A.nb,
                    T(k, k), T.mb,
                    A(m, k), A.nb,
                    work, T.nb);
            }
            else {
                ss_cond_wait(k, n, k);
                ss_cond_wait(m, n, k-1);
                CORE_zssmlq(
                    PlasmaRight, PlasmaConjTrans,
                    A.nb,
                    n == A.nt-1 ? A.n-n*A.nb : A.nb,
                    m == A.mt-1 ? A.m-m*A.nb : A.nb,
                    T.mb,
                    A.nb,
                    A(m, k), A.nb,
                    A(m, n), A.nb,
                    A(k, n), A.nb,
                    T(k, n), T.mb,
                    work, T.nb);
                ss_cond_set(m, n, k);
            }
        }
        m = next_m;
        n = next_n;
        k = next_k;
    }
    plasma_private_free(plasma, work);
    plasma_private_free(plasma, tau);
    ss_finalize();
}

/***************************************************************************//**
 *  Parallel tile LQ factorization - dynamic scheduling
 **/
void plasma_pzgelqf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request)
{
    int k, m, n;
    plasma_context_t *plasma;
    PLASMA_enum plasma_right = PlasmaRight;
    PLASMA_enum plasma_conj_trans = PlasmaConjTrans;
    int temp1, temp2, temp3;
    Quark_Task_Flags task_flags = Quark_Task_Flags_Initializer;

    plasma = plasma_context_self();
    if (sequence->status != PLASMA_SUCCESS)
        return;
    QUARK_Task_Flag_Set(&task_flags, TASK_SEQUENCE, (intptr_t)sequence->quark_sequence);

    for (k = 0; k < min(A.mt, A.nt); k++)
    {
        temp1 = A.m-k*A.nb;
        temp2 = A.n-k*A.nb;
        QUARK_Insert_Task(plasma->quark, CORE_zgelqt_quark, &task_flags,
            sizeof(int),                          k == A.mt-1 ? &temp1 : &A.nb, VALUE,
            sizeof(int),                          k == A.nt-1 ? &temp2 : &A.nb, VALUE,
            sizeof(int),                          &T.mb,                        VALUE,
            sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, k),                          INOUT | LOCALITY,
            sizeof(int),                          &A.nb,                        VALUE,
            sizeof(PLASMA_Complex64_t)*T.mb*T.nb, T(k, k),                          OUTPUT,
            sizeof(int),                          &T.mb,                        VALUE,
            sizeof(PLASMA_Complex64_t)*A.nb,      NULL,                             SCRATCH,
            sizeof(PLASMA_Complex64_t)*T.mb*T.nb, NULL,                             SCRATCH,
            0);

        for (m = k+1; m < A.mt; m++)
        {
            temp1 = A.m-m*A.nb;
            temp2 = A.n-k*A.nb;
            temp3 = A.n-k*A.nb;
            QUARK_Insert_Task(plasma->quark, CORE_zunmlq_quark, &task_flags,
                sizeof(PLASMA_enum),                  &plasma_right,                VALUE,
                sizeof(PLASMA_enum),                  &plasma_conj_trans,           VALUE,
                sizeof(int),                          m == A.mt-1 ? &temp1 : &A.nb, VALUE,
                sizeof(int),                          k == A.nt-1 ? &temp2 : &A.nb, VALUE,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(int),                          k == A.nt-1 ? &temp3 : &A.nb, VALUE,
                sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, k),                          INPUT,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*T.mb*T.nb, T(k, k),                          INPUT,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(m, k),                          INOUT | LOCALITY,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*T.mb*T.nb, NULL,                             SCRATCH,
                sizeof(int),                          &T.nb,                        VALUE,
                0);
        }

        for (n = k+1; n < A.nt; n++)
        {
            temp1 = A.m-k*A.nb;
            temp2 = A.n-n*A.nb;
            QUARK_Insert_Task(plasma->quark, CORE_ztslqt_quark, &task_flags,
                sizeof(int),                          k == A.mt-1 ? &temp1 : &A.nb, VALUE,
                sizeof(int),                          n == A.nt-1 ? &temp2 : &A.nb, VALUE,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, k),                          INOUT | LOCALITY,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, n),                          INOUT,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*T.mb*T.nb, T(k, n),                          OUTPUT,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex64_t)*A.nb,      NULL,                             SCRATCH,
                sizeof(PLASMA_Complex64_t)*T.mb*T.nb, NULL,                             SCRATCH,
                0);

            for (m = k+1; m < A.mt; m++)
            {
                temp1 = A.n-n*A.nb;
                temp2 = A.m-m*A.nb;
                QUARK_Insert_Task(plasma->quark, CORE_zssmlq_quark, &task_flags,
                    sizeof(PLASMA_enum),                  &plasma_right,                VALUE,
                    sizeof(PLASMA_enum),                  &plasma_conj_trans,           VALUE,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(int),                          n == A.nt-1 ? &temp1 : &A.nb, VALUE,
                    sizeof(int),                          m == A.mt-1 ? &temp2 : &A.nb, VALUE,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(m, k),                          INOUT | LOCALITY,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(m, n),                          INOUT,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex64_t)*A.mb*A.nb, A(k, n),                          INPUT,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex64_t)*T.mb*T.nb, T(k, n),                          INPUT,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(PLASMA_Complex64_t)*T.mb*T.nb, NULL,                             SCRATCH,
                    sizeof(int),                          &T.nb,                        VALUE,
                    0);
            }
        }
    }
}
