/**
 *
 * @file psormqr.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Hatem Ltaief
 * @author Jakub Kurzak
 * @date 2009-11-15
 *
 **/
#include "common.h"

#define A(m,n) BLKADDR(A, float, m, n)
#define B(m,n) BLKADDR(B, float, m, n)
#define T(m,n) BLKADDR(T, float, m, n)
/***************************************************************************//**
 *  Parallel application of Q using tile V - QR factorization - static scheduling
 **/
void plasma_psormqr(plasma_context_t *plasma)
{
    PLASMA_desc A;
    PLASMA_desc B;
    PLASMA_desc T;
    PLASMA_sequence *sequence;
    PLASMA_request *request;

    int k, m, n;
    int next_k;
    int next_m;
    int next_n;
    float *work;

    plasma_unpack_args_5(A, B, T, sequence, request);
    if (sequence->status != PLASMA_SUCCESS)
        return;

    work = (float*)plasma_private_alloc(plasma, T.mb*T.nb, T.dtyp);
    ss_init(B.mt, B.nt, -1);

    k = 0;
    n = PLASMA_RANK;
    while (n >= B.nt) {
        k++;
        n = n-B.nt;
    }
    m = k;

    while (k < min(A.mt, A.nt) && n < B.nt) {
        next_n = n;
        next_m = m;
        next_k = k;

        next_m++;
        if (next_m == A.mt) {
            next_n += PLASMA_SIZE;
            while (next_n >= B.nt && next_k < min(A.mt, A.nt)) {
                next_k++;
                next_n = next_n-B.nt;
            }
            next_m = next_k;
        }

        if (m == k) {
            ss_cond_wait(k, n, k-1);
            CORE_sormqr(
                PlasmaLeft, PlasmaTrans,
                k == A.mt-1 ? A.m-k*A.nb : A.nb,
                n == B.nt-1 ? B.n-n*B.nb : B.nb,
                T.mb,
                k == min(A.mt, A.nt)-1 ? min(A.m, A.n)-k*A.nb : A.nb,
                A(k, k), A.nb,
                T(k, k), T.mb,
                B(k, n), B.nb,
                work, T.nb);
            ss_cond_set(k, n, k);
        }
        else {
            ss_cond_wait(m, n, k-1);
            CORE_sssmqr(
                PlasmaLeft, PlasmaTrans,
                A.nb,
                m == A.mt-1 ? A.m-m*A.nb : A.nb,
                n == B.nt-1 ? B.n-n*B.nb : B.nb,
                T.mb,
                k == A.nt-1 ? A.n-k*A.nb : A.nb,
                B(k, n), B.nb,
                B(m, n), B.nb,
                A(m, k), A.nb,
                T(m, k), T.mb,
                work, T.mb);
            ss_cond_set(m, n, k);
        }
        n = next_n;
        m = next_m;
        k = next_k;
    }
    plasma_private_free(plasma, work);
    ss_finalize();
}

/***************************************************************************//**
 *  Parallel application of Q using tile V - QR factorization - dynamic scheduling
 **/
void plasma_psormqr_quark(PLASMA_desc A, PLASMA_desc B, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request)
{
    int k, m, n;
    plasma_context_t *plasma;
    PLASMA_enum plasma_left = PlasmaLeft;
    PLASMA_enum plasma__trans = PlasmaTrans;
    int temp1, temp2, temp3;
    Quark_Task_Flags task_flags = Quark_Task_Flags_Initializer;

    plasma = plasma_context_self();
    if (sequence->status != PLASMA_SUCCESS)
        return;
    QUARK_Task_Flag_Set(&task_flags, TASK_SEQUENCE, (intptr_t)sequence->quark_sequence);

    for (k = 0; k < min(A.mt, A.nt); k++)
    {
        for (n = 0; n < B.nt; n++) {
            temp1 = A.m-k*A.nb;
            temp2 = B.n-n*B.nb;
            temp3 = min(A.m, A.n)-k*A.nb;
            QUARK_Insert_Task(plasma->quark, CORE_sormqr_quark, &task_flags,
                sizeof(PLASMA_enum),                  &plasma_left,                            VALUE,
                sizeof(PLASMA_enum),                  &plasma__trans,                      VALUE,
                sizeof(int),                          k == A.mt-1 ? &temp1 : &A.nb,            VALUE,
                sizeof(int),                          n == B.nt-1 ? &temp2 : &B.nb,            VALUE,
                sizeof(int),                          &T.mb,                                   VALUE,
                sizeof(int),                          k == min(A.mt, A.nt)-1 ? &temp3 : &A.nb, VALUE,
                sizeof(float)*A.mb*A.nb, A(k, k),                                     INPUT,
                sizeof(int),                          &A.nb,                                   VALUE,
                sizeof(float)*T.mb*T.nb, T(k, k),                                     INPUT,
                sizeof(int),                          &T.mb,                                   VALUE,
                sizeof(float)*A.mb*A.nb, B(k, n),                                     INOUT | LOCALITY,
                sizeof(int),                          &B.nb,                                   VALUE,
                sizeof(float)*T.mb*T.nb, NULL,                                        SCRATCH,
                sizeof(int),                          &T.nb,                                   VALUE,
                0);
        }
        for (m = k+1; m < A.mt; m++)
        {
            for (n = 0; n < B.nt; n++)
            {
                temp1 = A.m-m*A.nb;
                temp2 = B.n-n*B.nb;
                temp3 = A.n-k*A.nb;
                QUARK_Insert_Task(plasma->quark, CORE_sssmqr_quark, &task_flags,
                    sizeof(PLASMA_enum),                  &plasma_left,                 VALUE,
                    sizeof(PLASMA_enum),                  &plasma__trans,           VALUE,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(int),                          m == A.mt-1 ? &temp1 : &A.nb, VALUE,
                    sizeof(int),                          n == B.nt-1 ? &temp2 : &B.nb, VALUE,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(int),                          k == A.nt-1 ? &temp3 : &A.nb, VALUE,
                    sizeof(float)*A.mb*A.nb, B(k, n),                          INOUT | LOCALITY,
                    sizeof(int),                          &B.nb,                        VALUE,
                    sizeof(float)*A.mb*A.nb, B(m, n),                          INOUT,
                    sizeof(int),                          &B.nb,                        VALUE,
                    sizeof(float)*A.mb*A.nb, A(m, k),                          INPUT,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(float)*T.mb*T.nb, T(m, k),                          INPUT,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(float)*T.mb*T.nb, NULL,                             SCRATCH,
                    sizeof(int),                          &T.mb,                        VALUE,
                    0);
            }
        }
    }
}
