/**
 *
 * @file pcgeqrf.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Jakub Kurzak
 * @author Hatem Ltaief
 * @date 2009-11-15
 *
 **/
#include "common.h"

#define A(m,n) BLKADDR(A, PLASMA_Complex32_t, m, n)
#define T(m,n) BLKADDR(T, PLASMA_Complex32_t, m, n)
/***************************************************************************//**
 *  Parallel tile QR factorization - static scheduling
 **/
void plasma_pcgeqrf(plasma_context_t *plasma)
{
    PLASMA_desc A;
    PLASMA_desc T;
    PLASMA_sequence *sequence;
    PLASMA_request *request;

    int k, m, n;
    int next_k;
    int next_m;
    int next_n;
    PLASMA_Complex32_t *work, *tau;

    plasma_unpack_args_4(A, T, sequence, request);
    if (sequence->status != PLASMA_SUCCESS)
        return;
    work = (PLASMA_Complex32_t*)plasma_private_alloc(plasma, T.mb*T.nb, T.dtyp);
    tau  = (PLASMA_Complex32_t*)plasma_private_alloc(plasma, A.nb, A.dtyp);
    ss_init(A.mt, A.nt, -1);

    k = 0;
    n = PLASMA_RANK;
    while (n >= A.nt) {
        k++;
        n = n-A.nt+k;
    }
    m = k;

    while (k < min(A.mt, A.nt) && n < A.nt) {
        next_n = n;
        next_m = m;
        next_k = k;

        next_m++;
        if (next_m == A.mt) {
            next_n += PLASMA_SIZE;
            while (next_n >= A.nt && next_k < min(A.mt, A.nt)) {
                next_k++;
                next_n = next_n-A.nt+next_k;
            }
            next_m = next_k;
        }

        if (n == k) {
            if (m == k) {
                ss_cond_wait(k, k, k-1);
                CORE_cgeqrt(
                    k == A.mt-1 ? A.m-k*A.nb : A.nb,
                    k == A.nt-1 ? A.n-k*A.nb : A.nb,
                    T.mb,
                    A(k, k), A.nb,
                    T(k, k), T.mb,
                    tau, work);
                ss_cond_set(k, k, k);
            }
            else {
                ss_cond_wait(m, k, k-1);
                CORE_ctsqrt(
                    m == A.mt-1 ? A.m-m*A.nb : A.nb,
                    k == A.nt-1 ? A.n-k*A.nb : A.nb,
                    T.mb,
                    A(k, k), A.nb,
                    A(m, k), A.nb,
                    T(m, k), T.mb,
                    tau, work);
                ss_cond_set(m, k, k);
            }
        }
        else {
            if (m == k) {
                ss_cond_wait(k, k, k);
                ss_cond_wait(k, n, k-1);
                CORE_cunmqr(
                    PlasmaLeft, PlasmaConjTrans,
                    k == A.mt-1 ? A.m-k*A.nb : A.nb,
                    n == A.nt-1 ? A.n-n*A.nb : A.nb,
                    T.mb,
                    k == A.mt-1 ? A.m-k*A.nb : A.nb,
                    A(k, k), A.nb,
                    T(k, k), T.mb,
                    A(k, n), A.nb,
                    work, T.nb);
            }
            else {
                ss_cond_wait(m, k, k);
                ss_cond_wait(m, n, k-1);
                CORE_cssmqr(
                    PlasmaLeft, PlasmaConjTrans,
                    A.nb,
                    m == A.mt-1 ? A.m-m*A.nb : A.nb,
                    n == A.nt-1 ? A.n-n*A.nb : A.nb,
                    T.mb,
                    A.nb,
                    A(k, n), A.nb,
                    A(m, n), A.nb,
                    A(m, k), A.nb,
                    T(m, k), T.mb,
                    work, T.mb);
                ss_cond_set(m, n, k);
            }
        }
        n = next_n;
        m = next_m;
        k = next_k;
    }
    plasma_private_free(plasma, work);
    plasma_private_free(plasma, tau);
    ss_finalize();
}

/***************************************************************************//**
 *  Parallel tile QR factorization - dynamic scheduling
 **/
void plasma_pcgeqrf_quark(PLASMA_desc A, PLASMA_desc T, PLASMA_sequence *sequence, PLASMA_request *request)
{
    int k, m, n;
    plasma_context_t *plasma;
    PLASMA_enum plasma_left = PlasmaLeft;
    PLASMA_enum plasma_conjf_trans = PlasmaConjTrans;
    int temp1, temp2, temp3;
    Quark_Task_Flags task_flags = Quark_Task_Flags_Initializer;

    plasma = plasma_context_self();
    if (sequence->status != PLASMA_SUCCESS)
        return;
    QUARK_Task_Flag_Set(&task_flags, TASK_SEQUENCE, (intptr_t)sequence->quark_sequence);

    for (k = 0; k < min(A.mt, A.nt); k++)
    {
        temp1 = A.m-k*A.nb;
        temp2 = A.n-k*A.nb;
        QUARK_Insert_Task(plasma->quark, CORE_cgeqrt_quark, &task_flags,
            sizeof(int),                          k == A.mt-1 ? &temp1 : &A.nb, VALUE,
            sizeof(int),                          k == A.nt-1 ? &temp2 : &A.nb, VALUE,
            sizeof(int),                          &T.mb,                        VALUE,
            sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(k, k),                          INOUT | LOCALITY,
            sizeof(int),                          &A.nb,                        VALUE,
            sizeof(PLASMA_Complex32_t)*T.mb*T.nb, T(k, k),                          OUTPUT,
            sizeof(int),                          &T.mb,                        VALUE,
            sizeof(PLASMA_Complex32_t)*A.nb,      NULL,                             SCRATCH,
            sizeof(PLASMA_Complex32_t)*T.mb*T.nb, NULL,                             SCRATCH,
            0);

        for (n = k+1; n < A.nt; n++)
        {
            temp1 = A.m-k*A.nb;
            temp2 = A.n-n*A.nb;
            temp3 = A.m-k*A.nb;
            QUARK_Insert_Task(plasma->quark, CORE_cunmqr_quark, &task_flags,
                sizeof(PLASMA_enum),                  &plasma_left,                 VALUE,
                sizeof(PLASMA_enum),                  &plasma_conjf_trans,           VALUE,
                sizeof(int),                          k == A.mt-1 ? &temp1 : &A.nb, VALUE,
                sizeof(int),                          n == A.nt-1 ? &temp2 : &A.nb, VALUE,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(int),                          k == A.mt-1 ? &temp3 : &A.nb, VALUE,
                sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(k, k),                          INPUT,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*T.mb*T.nb, T(k, k),                          INPUT,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(k, n),                          INOUT | LOCALITY,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*T.mb*T.nb, NULL,                             SCRATCH,
                sizeof(int),                          &T.nb,                        VALUE,
                0);
        }

        for (m = k+1; m < A.mt; m++)
        {
            temp1 = A.m-m*A.nb;
            temp2 = A.n-k*A.nb;
            QUARK_Insert_Task(plasma->quark, CORE_ctsqrt_quark, &task_flags,
                sizeof(int),                          m == A.mt-1 ? &temp1 : &A.nb, VALUE,
                sizeof(int),                          k == A.nt-1 ? &temp2 : &A.nb, VALUE,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(k, k),                          INOUT | LOCALITY,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(m, k),                          INOUT,
                sizeof(int),                          &A.nb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*T.mb*T.nb, T(m, k),                          OUTPUT,
                sizeof(int),                          &T.mb,                        VALUE,
                sizeof(PLASMA_Complex32_t)*A.nb,      NULL,                             SCRATCH,
                sizeof(PLASMA_Complex32_t)*T.mb*T.nb, NULL,                             SCRATCH,
                0);

            for (n = k+1; n < A.nt; n++)
            {
                temp1 = A.m-m*A.nb;
                temp2 = A.n-n*A.nb;
                QUARK_Insert_Task(plasma->quark, CORE_cssmqr_quark, &task_flags,
                    sizeof(PLASMA_enum),                  &plasma_left,                 VALUE,
                    sizeof(PLASMA_enum),                  &plasma_conjf_trans,           VALUE,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(int),                          m == A.mt-1 ? &temp1 : &A.nb, VALUE,
                    sizeof(int),                          n == A.nt-1 ? &temp2 : &A.nb, VALUE,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(k, n),                          INOUT | LOCALITY,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(m, n),                          INOUT,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex32_t)*A.mb*A.nb, A(m, k),                          INPUT,
                    sizeof(int),                          &A.nb,                        VALUE,
                    sizeof(PLASMA_Complex32_t)*T.mb*T.nb, T(m, k),                          INPUT,
                    sizeof(int),                          &T.mb,                        VALUE,
                    sizeof(PLASMA_Complex32_t)*T.mb*T.nb, NULL,                             SCRATCH,
                    sizeof(int),                          &T.mb,                        VALUE,
                    0);
            }
        }
    }
}
