/**
 *
 * @file pcgemm.c
 *
 *  PLASMA auxiliary routines
 *  PLASMA is a software package provided by Univ. of Tennessee,
 *  Univ. of California Berkeley and Univ. of Colorado Denver
 *
 * @version 2.2.0
 * @author Emmanuel Agullo
 * @author Mathieu Faverge
 * @date 2009-11-15
 *
 **/
#include "common.h"

#define A(m, n) BLKADDR(A, PLASMA_Complex32_t, m, n)
#define B(m, n) BLKADDR(B, PLASMA_Complex32_t, m, n)
#define C(m, n) BLKADDR(C, PLASMA_Complex32_t, m, n)
/***************************************************************************//**
 *  Parallel tile CGEMM matrix-matrix multiplication - static scheduling
 **/
void plasma_pcgemm(plasma_context_t *plasma)
{
    PLASMA_enum transA;
    PLASMA_enum transB;
    PLASMA_Complex32_t alpha;
    PLASMA_desc A;
    PLASMA_desc B;
    PLASMA_Complex32_t beta;
    PLASMA_desc C;
    PLASMA_sequence *sequence;
    PLASMA_request *request;

    PLASMA_Complex32_t *lA, *lB, *lC;
    int K, X, Y;
    int k, m, n;
    int kt, Anb, An;
    int next_m;
    int next_n;

    plasma_unpack_args_9(transA, transB, alpha, A, B, beta, C, sequence, request);
    if (sequence->status != PLASMA_SUCCESS)
        return;

    if (transA == PlasmaNoTrans) {
        Anb = A.nb;
        An  = A.n;
        kt  = A.nt;
    }
    else {
        Anb = A.mb;
        An  = A.m;
        kt  = A.mt;
    }

    n = 0;
    m = PLASMA_RANK;
    while (m >= C.mt && n < C.nt) {
        n++;
        m = m-C.mt;
    }

    while (n < C.nt) {
        next_m = m;
        next_n = n;

        next_m += PLASMA_SIZE;
        while (next_m >= C.mt && next_n < C.nt) {
            next_n++;
            next_m = next_m - C.mt;
        }

        X = m == C.mt-1 ? C.m - m*C.mb : C.mb;
        Y = n == C.nt-1 ? C.n - n*C.nb : C.nb;
        lC = C(m, n);

        for (k = 0 ; k < kt ; k++) {
            K = k == kt-1 ? An - k*Anb : Anb;

            lA = (transA == PlasmaNoTrans) ? A(m, k) : A(k, m);
            lB = (transB == PlasmaNoTrans) ? B(k, n) : B(n, k);

            CORE_cgemm( transA, transB, X, Y, K,
                        alpha, lA, A.mb,
                        lB, B.mb,
                        k == 0 ? beta : ((PLASMA_Complex32_t) 1.0),
                        lC, C.mb);
        }
        m = next_m;
        n = next_n;
    }
}

/***************************************************************************//**
 *  Parallel tile CGEMM matrix-matrix multiplication - dynamic scheduling
 **/
void plasma_pcgemm_quark(PLASMA_enum transA, PLASMA_enum transB,
                          PLASMA_Complex32_t alpha, PLASMA_desc A, PLASMA_desc B,
                          PLASMA_Complex32_t beta,  PLASMA_desc C,
                          PLASMA_sequence *sequence, PLASMA_request *request)
{
    plasma_context_t *plasma;
    int X, Y, Z;
    int m, n, k;
    int K, Anb, An;
    PLASMA_Complex32_t lbeta;
    PLASMA_Complex32_t *lA, *lB, *lC;
    Quark_Task_Flags task_flags = Quark_Task_Flags_Initializer;

    plasma = plasma_context_self();
    if (sequence->status != PLASMA_SUCCESS)
        return;
    QUARK_Task_Flag_Set(&task_flags, TASK_SEQUENCE, (intptr_t)sequence->quark_sequence);

    if (transA == PlasmaNoTrans) {
        Anb = A.nb;
        An  = A.n;
        K   = A.nt;
    }
    else {
        Anb = A.mb;
        An  = A.m;
        K   = A.mt;
    }

    for (m = 0; m < C.mt; m++) {
        X = m == C.mt-1 ? C.m - m * C.mb : C.mb;

        for (n = 0; n < C.nt; n++) {
            Y = n == C.nt-1 ? C.n - n * C.nb : C.nb;
            lC = C(m, n);

            for (k = 0; k < K; k++) {
                Z = k == K-1 ? An - k * Anb : Anb;

                lbeta = k == 0 ? beta : ((PLASMA_Complex32_t) 1.0);
                lA = (transA == PlasmaNoTrans) ? A(m, k) : A(k, m);
                lB = (transB == PlasmaNoTrans) ? B(k, n) : B(n, k);

                QUARK_Insert_Task(plasma->quark, CORE_cgemm_quark, &task_flags,
                    sizeof(PLASMA_enum),                  &transA, VALUE,
                    sizeof(PLASMA_enum),                  &transB, VALUE,
                    sizeof(int),                          &X,      VALUE,
                    sizeof(int),                          &Y,      VALUE,
                    sizeof(int),                          &Z,      VALUE,
                    sizeof(PLASMA_Complex32_t),           &alpha,  VALUE,
                    sizeof(PLASMA_Complex32_t)*A.mb*A.nb, lA,      INPUT,
                    sizeof(int),                          &A.mb,   VALUE,
                    sizeof(PLASMA_Complex32_t)*B.mb*B.nb, lB,      INPUT,
                    sizeof(int),                          &B.mb,   VALUE,
                    sizeof(PLASMA_Complex32_t),           &lbeta,  VALUE,
                    sizeof(PLASMA_Complex32_t)*C.mb*C.nb, lC,      INOUT | LOCALITY,
                    sizeof(int),                          &C.mb,   VALUE,
                    0);
            }
        }
    }
}
