/*
 blas.c
 */

#include <math.h>

#include <blas.h>

void
BLAS_error(char *rname, int err, int val, int x) {
  fprintf( stderr, "%s %d %d %d\n", rname, err, val, x );
  abort();
}

void
BLAS_dnorm(enum blas_norm_type norm, int n, const double *x, int incx, double *res) {
  char rname[] = "BLAS_dnorm";
  double xnorm, v;
  int i;

  if (norm == blas_inf_norm) {
    xnorm = 0.0;
    for (i = 0; i < n; ++i) {
      v = fabs(x[0]);
      if (v > xnorm) xnorm = v;
      x += incx;
    }
    *res = xnorm;
  } else {
    BLAS_error( rname, -2, norm, 0 );
    return;
  }
}

void
BLAS_dge_norm(enum blas_order_type order, enum blas_norm_type norm,
    int m, int n, const double *a, int lda, double *res) {
  int i, j;
  double anorm, v;
  char rname[] = "BLAS_dge_norm";

  if (order != blas_colmajor) BLAS_error( rname, -1, order, 0 );

  if (norm == blas_frobenius_norm) {
    anorm = 0.0;
    for (j = n; j; --j) {
      for (i = m; i; --i) {
        v = a[0];
        anorm += v * v;
        a++;
      }
      a += lda - m;
    }
    anorm = sqrt( anorm );
  } else if (norm == blas_inf_norm) {
    anorm = 0.0;
    for (i = 0; i < m; ++i) {
      v = 0.0;
      for (j = 0; j < n; ++j) {
        v += fabs( a[i + j * lda] );
      }
      if (v > anorm)
        anorm = v;
    }
  } else {
    BLAS_error( rname, -2, norm, 0 );
    return;
  }

  if (res) *res = anorm;
}

static double
blas_pow_di(double x, int n) {
  double rv = 1.0;

  if (n < 0) {
    n = -n;
    x = 1.0 / x;
  }

  for (; n; n >>= 1, x *= x) {
    if (n & 1)
      rv *= x;
  }

  return rv;
}

double
BLAS_dfpinfo(enum blas_cmach_type cmach) {
  double eps = 1.0, r = 1.0, o = 1.0, b = 2.0;
/* single precision IEEE 754:
  int t = 24, l = 128,  m = -125;
*/
  int t = 53, l = 1024, m = -1021;
  char rname[] = "BLAS_dfpinfo";

  /* for (i = 0; i < t; ++i) eps *= half; */
  eps = blas_pow_di( b, -t );
  /* for (i = 0; i >= m; --i) r *= half; */
  r = blas_pow_di( b, m-1 );

  o -= eps;
  /* for (i = 0; i < l; ++i) o *= b; */
  o = (o * blas_pow_di( b, l-1 )) * b;

  switch (cmach) {
    case blas_eps: return eps;
    case blas_sfmin: return r;
    default:
      BLAS_error( rname, -1, cmach, 0 );
      break;
  }
  return 0.0;
}
