/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgetrs - Solves a system of linear equations A * X = B, with a general N-by-N matrix A
// using the tile LU factorization computed by PLASMA_zgetrf.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// trans    PLASMA_enum (IN)
//          Intended to specify the the form of the system of equations:
//          = PlasmaNoTrans:   A * X = B     (No transpose)
//          = PlasmaTrans:     A**T * X = B  (Transpose)
//          = PlasmaConjTrans: A**H * X = B  (Conjugate transpose)
//          Currently only PlasmaNoTrans is supported.
//
// N        int (IN)
//          The order of the matrix A.  N >= 0.
//
// NRHS     int (IN)
//          The number of right hand sides, i.e., the number of columns of the matrix B.
//          NRHS >= 0.
//
// A        PLASMA_Complex64_t* (IN)
//          The tile factors L and U from the factorization, computed by PLASMA_zgetrf.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).
//
// L        PLASMA_Complex64_t* (IN)
//          Auxiliary factorization data, related to the tile L factor, computed by PLASMA_zgetrf.
//
// IPIV     int* (IN)
//          The pivot indices from PLASMA_zgetrf (not equivalent to LAPACK).
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the N-by-NRHS matrix of right hand side matrix B.
//          On exit, the solution matrix X.
//
// LDB      int (IN)
//          The leading dimension of the array B. LDB >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_zgetrs(PLASMA_enum trans, int N, int NRHS, PLASMA_Complex64_t *A, int LDA,
                  PLASMA_Complex64_t *L, int *IPIV, PLASMA_Complex64_t *B, int LDB)
{
    int NB, NT, NTRHS;
    int status;
    PLASMA_Complex64_t *Abdl;
    PLASMA_Complex64_t *Bbdl;
    PLASMA_Complex64_t *Lbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgetrs", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (trans != PlasmaNoTrans) {
        plasma_error("PLASMA_zgetrs", "only PlasmaNoTrans supported");
        return PLASMA_ERR_NOT_SUPPORTED;
    }
    if (N < 0) {
        plasma_error("PLASMA_zgetrs", "illegal value of N");
        return -2;
    }
    if (NRHS < 0) {
        plasma_error("PLASMA_zgetrs", "illegal value of NRHS");
        return -3;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_zgetrs", "illegal value of LDA");
        return -5;
    }
    if (LDB < max(1, N)) {
        plasma_error("PLASMA_zgetrs", "illegal value of LDB");
        return -9;
    }
    /* Quick return */
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_ZGESV, N, N, NRHS);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgetrs", "plasma_tune() failed");
        return status;
    }

    /* Set NT & NTRHS */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    NTRHS = (NRHS%NB==0) ? (NRHS/NB) : (NRHS/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    Lbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, NT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);
    Bbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, NT*NTRHS*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    if (Abdl == NULL || Lbdl == NULL || Bbdl == NULL) {
        plasma_error("PLASMA_zgetrs", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Lbdl);
        plasma_shared_free(plasma, Bbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    PLASMA_desc descB = plasma_desc_init(
        Bbdl, PlasmaComplexDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, NRHS, 0, 0, N, NRHS);

    PLASMA_desc descL = plasma_desc_init(
        Lbdl, PlasmaComplexDouble,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        N, N, 0, 0, N, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex64_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex64_t*, B,
        int, LDB,
        PLASMA_desc, descB);

    /* Receive L from the user */
    plasma_memcpy(Lbdl, L, NT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);

    /* Call the native interface */
    status = PLASMA_zgetrs_Tile(&descA, &descL, IPIV, &descB);

    if (status == PLASMA_SUCCESS)
        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            PLASMA_Complex64_t*, B,
            int, LDB);

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Lbdl);
    plasma_shared_free(plasma, Bbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgetrs_Tile - Solves a system of linear equations A * X = B, with a general N-by-N
// matrix A using the tile LU factorization computed by PLASMA_zgetrf.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        PLASMA_Complex64_t* (IN)
//          The tile factors L and U from the factorization, computed by PLASMA_zgetrf.
//
// L        PLASMA_Complex64_t* (IN)
//          Auxiliary factorization data, related to the tile L factor, computed by PLASMA_zgetrf.
//
// IPIV     int* (IN)
//          The pivot indices from PLASMA_zgetrf (not equivalent to LAPACK).
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the N-by-NRHS matrix of right hand side matrix B.
//          On exit, the solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_zgetrs_Tile(PLASMA_desc *A, PLASMA_desc *L, int *IPIV, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descL = *L;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgetrs_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgetrs_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descL) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgetrs_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgetrs_Tile", "invalid third descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb || descB.nb != descB.mb) {
        plasma_error("PLASMA_zgetrs_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_4(plasma_pztrsmpl,
        PLASMA_desc, descA,
        PLASMA_desc, descB,
        PLASMA_desc, descL,
        int*, IPIV);

    plasma_parallel_call_7(plasma_pztrsm,
        PLASMA_enum, PlasmaLeft,
        PLASMA_enum, PlasmaUpper,
        PLASMA_enum, PlasmaNoTrans,
        PLASMA_enum, PlasmaNonUnit,
        PLASMA_Complex64_t, 1.0,
        PLASMA_desc, descA,
        PLASMA_desc, descB);

    return PLASMA_SUCCESS;
}
