/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_strsmpl - Performs the forward substitution step of solving a system of linear equations
// after the tile LU factorization of the matrix.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// N        int (IN)
//          The order of the matrix A. N >= 0.
//
// NRHS     int (IN)
//          The number of right hand sides, i.e., the number of columns of the matrix B. NRHS >= 0.
//
// A        float* (IN)
//          The tile factor L from the factorization, computed by PLASMA_sgetrf.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).
//
// L        float* (IN)
//          Auxiliary factorization data, related to the tile L factor, computed by PLASMA_sgetrf.
//
// IPIV     int* (IN)
//          The pivot indices from PLASMA_sgetrf (not equivalent to LAPACK).
//
// B        float* (INOUT)
//          On entry, the N-by-NRHS right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.
//
// LDB      float* (IN)
//          The leading dimension of the array B. LDB >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_strsmpl(int N, int NRHS, float *A, int LDA, float *L,
                   int *IPIV, float *B, int LDB)
{
    int NB, NT, NTRHS;
    int status;
    float *Abdl;
    float *Bbdl;
    float *Lbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_strsmpl", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (N < 0) {
        plasma_error("PLASMA_strsmpl", "illegal value of N");
        return -1;
    }
    if (NRHS < 0) {
        plasma_error("PLASMA_strsmpl", "illegal value of NRHS");
        return -2;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_strsmpl", "illegal value of LDA");
        return -4;
    }
    if (LDB < max(1, N)) {
        plasma_error("PLASMA_strsmpl", "illegal value of LDB");
        return -8;
    }
    /* Quick return */
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_SGESV, N, N, NRHS);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_strsmpl", "plasma_tune() failed");
        return status;
    }

    /* Set Mt, NT & NTRHS */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    NTRHS = (NRHS%NB==0) ? (NRHS/NB) : (NRHS/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (float *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaRealFloat);
    Lbdl = (float *)plasma_shared_alloc(plasma, NT*NT*PLASMA_IBNBSIZE, PlasmaRealFloat);
    Bbdl = (float *)plasma_shared_alloc(plasma, NT*NTRHS*PLASMA_NBNBSIZE, PlasmaRealFloat);
    if (Abdl == NULL || Lbdl == NULL || Bbdl == NULL) {
        plasma_error("PLASMA_strsmpl", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Lbdl);
        plasma_shared_free(plasma, Bbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaRealFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    PLASMA_desc descB = plasma_desc_init(
        Bbdl, PlasmaRealFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, NRHS, 0, 0, N, NRHS);

    PLASMA_desc descL = plasma_desc_init(
        Lbdl, PlasmaRealFloat,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        N, N, 0, 0, N, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        float*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        float*, B,
        int, LDB,
        PLASMA_desc, descB);

    /* Receive L from the user */
    plasma_memcpy(Lbdl, L, NT*NT*PLASMA_IBNBSIZE, PlasmaRealFloat);

    /* Call the native interface */
    status = PLASMA_strsmpl_Tile(&descA, &descL, IPIV, &descB);

    if (status == PLASMA_SUCCESS)
        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            float*, B,
            int, LDB);

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Lbdl);
    plasma_shared_free(plasma, Bbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_strsmpl_Tile - Performs the forward substitution step of solving a system of linear
// equations after the tile LU factorization of the matrix.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        float* (IN)
//          The tile factor L from the factorization, computed by PLASMA_sgetrf.
//
// L        float* (IN)
//          Auxiliary factorization data, related to the tile L factor, computed by PLASMA_sgetrf.
//
// IPIV     int* (IN)
//          The pivot indices from PLASMA_sgetrf (not equivalent to LAPACK).
//
// B        float* (INOUT)
//          On entry, the N-by-NRHS right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_strsmpl_Tile(PLASMA_desc *A, PLASMA_desc *L, int *IPIV, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descL = *L;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_strsmpl_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_strsmpl_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descL) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_strsmpl_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_strsmpl_Tile", "invalid third descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb || descB.nb != descB.mb) {
        plasma_error("PLASMA_strsmpl_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_4(plasma_pstrsmpl,
        PLASMA_desc, descA,
        PLASMA_desc, descB,
        PLASMA_desc, descL,
        int*, IPIV);

    return PLASMA_SUCCESS;
}
