/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_sgetrf - Computes an LU factorization of a general M-by-N matrix A
// using the tile LU algorithm with partial tile pivoting with row interchanges.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A. N >= 0.
//
// A        float* (INOUT)
//          On entry, the M-by-N matrix to be factored.
//          On exit, the tile factors L and U from the factorization.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// L        float* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          required by PLASMA_sgetrs to solve the system of equations.
//
// IPIV     int* (OUT)
//          The pivot indices that define the permutations (not equivalent to LAPACK).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, and division by zero will occur
//               if it is used to solve a system of equations.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_sgetrf(int M, int N, float *A, int LDA, float *L, int *IPIV)
{
    int NB, MT, NT;
    int status;
    float *Abdl;
    float *Lbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_sgetrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (M < 0) {
        plasma_error("PLASMA_sgetrf", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_sgetrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_sgetrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_SGESV, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_sgetrf", "plasma_tune() failed");
        return status;
    }

    /* Set NT & NTRHS */
    NB = PLASMA_NB;
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (float *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaRealFloat);
    Lbdl = (float *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaRealFloat);
    if (Abdl == NULL || Lbdl == NULL) {
        plasma_error("PLASMA_sgetrf", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Lbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaRealFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        M, N, 0, 0, M, N);

    PLASMA_desc descL = plasma_desc_init(
        Lbdl, PlasmaRealFloat,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        M, N, 0, 0, M, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        float*, A,
        int, LDA,
        PLASMA_desc, descA);

    /* Call the native interface */
    status = PLASMA_sgetrf_Tile(&descA, &descL, IPIV);

    if (status == PLASMA_SUCCESS) {
        /* Return L to the user */
        plasma_memcpy(L, Lbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaRealFloat);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            float*, A,
            int, LDA);
    }
    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Lbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_sgetrf_Tile - Computes an LU factorization of a general M-by-N matrix A
// using the tile LU algorithm with partial tile pivoting with row interchanges.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        float* (INOUT)
//          On entry, the M-by-N matrix to be factored.
//          On exit, the tile factors L and U from the factorization.
//
// L        float* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          required by PLASMA_sgetrs to solve the system of equations.
//
// IPIV     int* (OUT)
//          The pivot indices that define the permutations (not equivalent to LAPACK).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, and division by zero will occur
//               if it is used to solve a system of equations.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_sgetrf_Tile(PLASMA_desc *A, PLASMA_desc *L, int *IPIV)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descL = *L;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_sgetrf_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_sgetrf_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descL) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_sgetrf_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb) {
        plasma_error("PLASMA_sgetrf_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;
*/
    /* Clear IPIV and Lbdl */
    plasma_memzero(IPIV, descA.mt*descA.nt*PLASMA_NB, PlasmaInteger);
    plasma_memzero(descL.mat, descL.mt*descL.nt*PLASMA_IBNBSIZE, PlasmaRealFloat);

    /* Set INFO to SUCCESS */
    PLASMA_INFO = PLASMA_SUCCESS;

    plasma_parallel_call_3(plasma_psgetrf,
        PLASMA_desc, descA,
        PLASMA_desc, descL,
        int*, IPIV);

    return PLASMA_INFO;
}
