/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_dgesv - Computes the solution to a system of linear equations A * X = B,
// where A is an N-by-N matrix and X and B are N-by-NRHS matrices.
// The tile LU decomposition with partial tile pivoting and row interchanges is used to factor A.
// The factored form of A is then used to solve the system of equations A * X = B.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// N        int (IN)
//          The number of linear equations, i.e., the order of the matrix A. N >= 0.
//
// NRHS     int (IN)
//          The number of right hand sides, i.e., the number of columns of the matrix B.
//          NRHS >= 0.
//
// A        double* (INOUT)
//          On entry, the N-by-N coefficient matrix A.
//          On exit, the tile L and U factors from the factorization (not equivalent to LAPACK).
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).
//
// L        double* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          necessary to solve the system of equations.
//
// IPIV     int* (OUT)
//          On exit, the pivot indices that define the permutations (not equivalent to LAPACK).
//
// B        double* (INOUT)
//          On entry, the N-by-NRHS matrix of right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.
//
// LDB      int (IN)
//          The leading dimension of the array B. LDB >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, so the solution could not be computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_dgesv(int N, int NRHS, double *A, int LDA, double *L, int *IPIV,
                 double *B, int LDB)
{
    int NB, NT, NTRHS;
    int status;
    double *Abdl;
    double *Bbdl;
    double *Lbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_error("PLASMA_dgesv", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (N < 0) {
        plasma_error("PLASMA_dgesv", "illegal value of N");
        return -1;
    }
    if (NRHS < 0) {
        plasma_error("PLASMA_dgesv", "illegal value of NRHS");
        return -2;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_dgesv", "illegal value of LDA");
        return -4;
    }
    if (LDB < max(1, N)) {
        plasma_error("PLASMA_dgesv", "illegal value of LDB");
        return -8;
    }
    /* Quick return */
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_DGESV, N, N, NRHS);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgesv", "plasma_tune() failed");
        return status;
    }

    /* Set NT & NTRHS */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    NTRHS = (NRHS%NB==0) ? (NRHS/NB) : (NRHS/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (double *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaRealDouble);
    Lbdl = (double *)plasma_shared_alloc(plasma, NT*NT*PLASMA_IBNBSIZE, PlasmaRealDouble);
    Bbdl = (double *)plasma_shared_alloc(plasma, NT*NTRHS*PLASMA_NBNBSIZE, PlasmaRealDouble);
    if (Abdl == NULL || Lbdl == NULL || Bbdl == NULL) {
        plasma_error("PLASMA_dgesv", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Lbdl);
        plasma_shared_free(plasma, Bbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaRealDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    PLASMA_desc descB = plasma_desc_init(
        Bbdl, PlasmaRealDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, NRHS, 0, 0, N, NRHS);

    PLASMA_desc descL = plasma_desc_init(
        Lbdl, PlasmaRealDouble,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        N, N, 0, 0, N, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        double*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        double*, B,
        int, LDB,
        PLASMA_desc, descB);

    /* Call the native interface */
    status = PLASMA_dgesv_Tile(&descA, &descL, IPIV, &descB);

    if (status == PLASMA_SUCCESS) {
        /* Return L to the user */
        plasma_memcpy(L, Lbdl, NT*NT*PLASMA_IBNBSIZE, PlasmaRealDouble);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            double*, A,
            int, LDA);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            double*, B,
            int, LDB);
    }
    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Lbdl);
    plasma_shared_free(plasma, Bbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_dgesv_Tile - Computes the solution to a system of linear equations A * X = B,
// where A is an N-by-N matrix and X and B are N-by-NRHS matrices.
// The tile LU decomposition with partial tile pivoting and row interchanges is used to factor A.
// The factored form of A is then used to solve the system of equations A * X = B.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        double* (INOUT)
//          On entry, the N-by-N coefficient matrix A.
//          On exit, the tile L and U factors from the factorization (not equivalent to LAPACK).
//
// L        double* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          necessary to solve the system of equations.
//
// IPIV     int* (OUT)
//          On exit, the pivot indices that define the permutations (not equivalent to LAPACK).
//
// B        double* (INOUT)
//          On entry, the N-by-NRHS matrix of right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, so the solution could not be computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_dgesv_Tile(PLASMA_desc *A, PLASMA_desc *L, int *IPIV, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descL = *L;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_dgesv_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgesv_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descL) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgesv_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgesv_Tile", "invalid third descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb || descB.nb != descB.mb) {
        plasma_error("PLASMA_dgesv_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    /* Clear IPIV and Lbdl */
    plasma_memzero(IPIV, descA.mt*descA.nt*PLASMA_NB, PlasmaInteger);
    plasma_memzero(descL.mat, descL.mt*descL.nt*PLASMA_IBNBSIZE, PlasmaRealDouble);

    /* Set INFO to ZERO */
    PLASMA_INFO = PLASMA_SUCCESS;

    plasma_parallel_call_3(plasma_pdgetrf,
        PLASMA_desc, descA,
        PLASMA_desc, descL,
        int*, IPIV);

    if (PLASMA_INFO == PLASMA_SUCCESS) {
        plasma_parallel_call_4(plasma_pdtrsmpl,
            PLASMA_desc, descA,
            PLASMA_desc, descB,
            PLASMA_desc, descL,
            int*, IPIV);

        plasma_parallel_call_7(plasma_pdtrsm,
            PLASMA_enum, PlasmaLeft,
            PLASMA_enum, PlasmaUpper,
            PLASMA_enum, PlasmaNoTrans,
            PLASMA_enum, PlasmaNonUnit,
            double, 1.0,
            PLASMA_desc, descA,
            PLASMA_desc, descB);
    }
    return PLASMA_INFO;
}
