/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_dgeqrf - Computes the tile QR factorization of a complex M-by-N matrix A: A = Q * R.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A.  N >= 0.
//
// A        double* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit, the elements on and above the diagonal of the array contain the min(M,N)-by-N
//          upper trapezoidal matrix R (R is upper triangular if M >= N); the elements below the
//          diagonal represent the unitary matrix Q as a product of elementary reflectors stored
//          by tiles.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// T        double* (OUT)
//          On exit, auxiliary factorization data, required by PLASMA_dgeqrs to solve the system
//          of equations.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_dgeqrf(int M, int N, double *A, int LDA, double *T)
{
    int NB, MT, NT;
    int status;
    double *Abdl;
    double *Tbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_dgeqrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (M < 0) {
        plasma_error("PLASMA_dgeqrf", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_dgeqrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_dgeqrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_DGELS, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgeqrf", "plasma_tune() failed");
        return status;
    }

    /* Set MT & NT */
    NB = PLASMA_NB;
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (double *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaRealDouble);
    Tbdl = (double *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaRealDouble);
    if (Abdl == NULL || Tbdl == NULL) {
        plasma_error("PLASMA_dgeqrf", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Tbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaRealDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        M, N, 0, 0, M, N);

    PLASMA_desc descT = plasma_desc_init(
        Tbdl, PlasmaRealDouble,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        M, N, 0, 0, M, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        double*, A,
        int, LDA,
        PLASMA_desc, descA);

    /* Call the native interface */
    status = PLASMA_dgeqrf_Tile(&descA, &descT);

    if (status == PLASMA_SUCCESS) {
        /* Return T to the user */
        plasma_memcpy(T, Tbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaRealDouble);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            double*, A,
            int, LDA);
    }
    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Tbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_dgeqrf_Tile - Computes the tile QR factorization of a complex M-by-N matrix A: A = Q * R.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        double* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit, the elements on and above the diagonal of the array contain the min(M,N)-by-N
//          upper trapezoidal matrix R (R is upper triangular if M >= N); the elements below the
//          diagonal represent the unitary matrix Q as a product of elementary reflectors stored
//          by tiles.
//
// T        double* (OUT)
//          On exit, auxiliary factorization data, required by PLASMA_dgeqrs to solve the system
//          of equations.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_dgeqrf_Tile(PLASMA_desc *A, PLASMA_desc *T)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descT = *T;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_error("PLASMA_dgeqrf_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgeqrf_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descT) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_dgeqrf_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb) {
        plasma_error("PLASMA_dgeqrf_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_2(plasma_pdgeqrf,
        PLASMA_desc, descA,
        PLASMA_desc, descT);

    return PLASMA_SUCCESS;
}
