/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_ctrsm - Computes triangular solve A*X = B or X*A = B

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// side     PLASMA_enum (IN)
//          Specifies whether A appears on the left or on the right of X:
//          = PlasmaLeft:  A*X = B
//          = PlasmaRight: X*A = B
//
// uplo     PLASMA_enum (IN)
//          Specifies whether the matrix A is upper triangular or lower triangular:
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// transA   PLASMA_enum (IN)
//          Specifies whether the matrix A is transposed, not transposed or conjugate transposed:
//          = PlasmaNoTrans:   A is transposed;
//          = PlasmaTrans:     A is not transposed;
//          = PlasmaConjTrans: A is conjugate transposed.
//
// diag     PLASMA_enum (IN)
//          Specifies whether or not A is unit triangular:
//          = PlasmaNonUnit: A is non unit;
//          = PlasmaUnit:    A us unit.
//
// N        int (IN)
//          The order of the matrix A. N >= 0.
//
// NRHS     int (IN)
//          The number of right hand sides, i.e., the number of columns of the matrix B. NRHS >= 0.
//
// A        PLASMA_Complex32_t* (IN)
//          The triangular matrix A. If uplo = PlasmaUpper, the leading N-by-N upper triangular
//          part of the array A contains the upper triangular matrix, and the strictly lower
//          triangular part of A is not referenced. If uplo = PlasmaLower, the leading N-by-N
//          lower triangular part of the array A contains the lower triangular matrix, and the
//          strictly upper triangular part of A is not referenced. If diag = PlasmaUnit, the
//          diagonal elements of A are also not referenced and are assumed to be 1.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).
//
// B        PLASMA_Complex32_t* (INOUT)
//          On entry, the N-by-NRHS right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.
//
// LDB      PLASMA_Complex32_t* (IN)
//          The leading dimension of the array B. LDB >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_ctrsm(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, int N,
                 int NRHS, PLASMA_Complex32_t *A, int LDA, PLASMA_Complex32_t *B, int LDB)
{
    int NB, NT, NTRHS;
    int status;
    PLASMA_Complex32_t *Abdl;
    PLASMA_Complex32_t *Bbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_ctrsm", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (side != PlasmaLeft && side != PlasmaRight) {
        plasma_error("PLASMA_ctrsm", "illegal value of side");
        return -1;
    }
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_ctrsm", "illegal value of uplo");
        return -2;
    }
    if (transA != PlasmaConjTrans && transA != PlasmaNoTrans) {
        plasma_error("PLASMA_ctrsm", "illegal value of transA");
        return -3;
    }
    if (diag != PlasmaUnit && diag != PlasmaNonUnit) {
        plasma_error("PLASMA_ctrsm", "illegal value of diag");
        return -4;
    }
    if (N < 0) {
        plasma_error("PLASMA_ctrsm", "illegal value of N");
        return -5;
    }
    if (NRHS < 0) {
        plasma_error("PLASMA_ctrsm", "illegal value of NRHS");
        return -6;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_ctrsm", "illegal value of LDA");
        return -8;
    }
    if (LDB < max(1, N)) {
        plasma_error("PLASMA_ctrsm", "illegal value of LDB");
        return -10;
    }
    /* Quick return */
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_CPOSV, N, N, NRHS);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_ctrsm", "plasma_tune() failed");
        return status;
    }

    /* Set NT & NTRHS */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    NTRHS = (NRHS%NB==0) ? (NRHS/NB) : (NRHS/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex32_t *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaComplexFloat);
    Bbdl = (PLASMA_Complex32_t *)plasma_shared_alloc(plasma, NT*NTRHS*PLASMA_NBNBSIZE, PlasmaComplexFloat);
    if (Abdl == NULL || Bbdl == NULL) {
        plasma_error("PLASMA_ctrsm", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Bbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    PLASMA_desc descB = plasma_desc_init(
        Bbdl, PlasmaComplexFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, NRHS, 0, 0, N, NRHS);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex32_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex32_t*, B,
        int, LDB,
        PLASMA_desc, descB);

    /* Call the native interface */
    status = PLASMA_ctrsm_Tile(PlasmaLeft, uplo, transA, diag, &descA, &descB);

    if (status == PLASMA_SUCCESS)
        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            PLASMA_Complex32_t*, B,
            int, LDB);

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Bbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_ctrsm_Tile - Computes triangular solve A*X = B or X*A = B
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// side     PLASMA_enum (IN)
//          Specifies whether A appears on the left or on the right of X:
//          = PlasmaLeft:  A*X = B
//          = PlasmaRight: X*A = B
//
// uplo     PLASMA_enum (IN)
//          Specifies whether the matrix A is upper triangular or lower triangular:
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// transA   PLASMA_enum (IN)
//          Specifies whether the matrix A is transposed, not transposed or conjugate transposed:
//          = PlasmaNoTrans:   A is transposed;
//          = PlasmaTrans:     A is not transposed;
//          = PlasmaConjTrans: A is conjugate transposed.
//
// diag     PLASMA_enum (IN)
//          Specifies whether or not A is unit triangular:
//          = PlasmaNonUnit: A is non unit;
//          = PlasmaUnit:    A us unit.
//
// A        PLASMA_Complex32_t* (IN)
//          The triangular matrix A. If uplo = PlasmaUpper, the leading N-by-N upper triangular
//          part of the array A contains the upper triangular matrix, and the strictly lower
//          triangular part of A is not referenced. If uplo = PlasmaLower, the leading N-by-N
//          lower triangular part of the array A contains the lower triangular matrix, and the
//          strictly upper triangular part of A is not referenced. If diag = PlasmaUnit, the
//          diagonal elements of A are also not referenced and are assumed to be 1.
//
// B        PLASMA_Complex32_t* (INOUT)
//          On entry, the N-by-NRHS right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_ctrsm_Tile(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag,
                      PLASMA_desc *A, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_ctrsm_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_ctrsm_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_ctrsm_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb || descB.nb != descB.mb) {
        plasma_error("PLASMA_ctrsm_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (side != PlasmaLeft && side != PlasmaRight) {
        plasma_error("PLASMA_ctrsm_Tile", "illegal value of side");
        return -1;
    }
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_ctrsm_Tile", "illegal value of uplo");
        return -2;
    }
    if (transA != PlasmaConjTrans && transA != PlasmaNoTrans) {
        plasma_error("PLASMA_ctrsm_Tile", "illegal value of transA");
        return -3;
    }
    if (diag != PlasmaUnit && diag != PlasmaNonUnit) {
        plasma_error("PLASMA_ctrsm_Tile", "illegal value of diag");
        return -4;
    }
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_7(plasma_pctrsm,
        PLASMA_enum, PlasmaLeft,
        PLASMA_enum, uplo,
        PLASMA_enum, transA,
        PLASMA_enum, diag,
        PLASMA_Complex32_t, 1.0,
        PLASMA_desc, descA,
        PLASMA_desc, descB);

    return PLASMA_SUCCESS;
}
