/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_cpotrf - Computes the Cholesky factorization of a symmetric positive definite
// (or Hermitian positive definite in the complex case) matrix A.
// The factorization has the form
//
//   A = U**H * U, if uplo = PlasmaUpper, or
//   A = L * L**H, if uplo =  PlasmaLower,
//
// where U is an upper triangular matrix and L is a lower triangular matrix.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// uplo     PLASMA_enum (IN)
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// N        int (IN)
//          The order of the matrix A. N >= 0.
//
// A        PLASMA_Complex32_t* (INOUT)
//          On entry, the symmetric positive definite (or Hermitian) matrix A.
//          If uplo = PlasmaUpper, the leading N-by-N upper triangular part of A
//          contains the upper triangular part of the matrix A, and the strictly lower triangular
//          part of A is not referenced.
//          If UPLO = 'L', the leading N-by-N lower triangular part of A contains the lower
//          triangular part of the matrix A, and the strictly upper triangular part of A is not
//          referenced.
//          On exit, if return value = 0, the factor U or L from the Cholesky factorization
//          A = U**H*U or A = L*L**H.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value
//          > 0: if i, the leading minor of order i of A is not positive definite, so the
//               factorization could not be completed, and the solution has not been computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_cpotrf(PLASMA_enum uplo, int N, PLASMA_Complex32_t *A, int LDA)
{
    int NB, NT;
    int status;
    PLASMA_Complex32_t *Abdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_cpotrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_cpotrf", "illegal value of uplo");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_cpotrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_cpotrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (max(N, 0) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_CPOSV, N, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_cpotrf", "plasma_tune() failed");
        return status;
    }

    /* Set NT */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex32_t *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaComplexFloat);
    if (Abdl == NULL) {
        plasma_error("PLASMA_cpotrf", "plasma_shared_alloc() failed");
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex32_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    /* Call the native interface */
    status = PLASMA_cpotrf_Tile(uplo, &descA);

    if (status == PLASMA_SUCCESS)
        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex32_t*, A,
            int, LDA);

    plasma_shared_free(plasma, Abdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_cpotrf_Tile - Computes the Cholesky factorization of a symmetric positive definite
// (or Hermitian positive definite in the complex case) matrix A.
// The factorization has the form
//
//   A = U**H * U, if uplo = PlasmaUpper, or
//   A = L * L**H, if uplo =  PlasmaLower,
//
// where U is an upper triangular matrix and L is a lower triangular matrix.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// uplo     PLASMA_enum (IN)
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// A        PLASMA_Complex32_t* (INOUT)
//          On entry, the symmetric positive definite (or Hermitian) matrix A.
//          If uplo = PlasmaUpper, the leading N-by-N upper triangular part of A
//          contains the upper triangular part of the matrix A, and the strictly lower triangular
//          part of A is not referenced.
//          If UPLO = 'L', the leading N-by-N lower triangular part of A contains the lower
//          triangular part of the matrix A, and the strictly upper triangular part of A is not
//          referenced.
//          On exit, if return value = 0, the factor U or L from the Cholesky factorization
//          A = U**H*U or A = L*L**H.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          > 0: if i, the leading minor of order i of A is not positive definite, so the
//               factorization could not be completed, and the solution has not been computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_cpotrf_Tile(PLASMA_enum uplo, PLASMA_desc *A)
{
    PLASMA_desc descA = *A;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_cpotrf_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_cpotrf_Tile", "invalid descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb) {
        plasma_error("PLASMA_cpotrf_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_cpotrf_Tile", "illegal value of uplo");
        return -1;
    }
    /* Quick return */
/*
    if (max(N, 0) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_2(plasma_pcpotrf,
        PLASMA_enum, uplo,
        PLASMA_desc, descA);

    return PLASMA_INFO;
}
