/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.1.0)                          ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"
#include "auxiliary.h"
#include "context.h"

#if defined( _WIN32 ) || defined( _WIN64 )
#include "plasmawinthread.h"
#else
#include <pthread.h>
#endif

#include <stdlib.h>

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Global data
// master threads context lookup table
plasma_context_map_t context_map[CONTEXTS_MAX];
// context lookup table access lock
pthread_mutex_t context_map_lock = PTHREAD_MUTEX_INITIALIZER;

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
// Create new context
plasma_context_t *plasma_context_create()
{
    plasma_context_t *plasma;

    plasma = (plasma_context_t*)malloc(sizeof(plasma_context_t));
    if (plasma == NULL) {
        plasma_fatal_error("plasma_context_create", "malloc() failed");
        return NULL;
    }

    pthread_mutex_init(&plasma->action_mutex, NULL);
    pthread_mutex_init(&context_map_lock, NULL);
    pthread_cond_init(&plasma->action_condt, NULL);
    plasma->action = PLASMA_ACT_STAND_BY;
    plasma->parallel_func_ptr = NULL;

    /* These initializations are just in case the user
       disables autotuning and does not set nb and ib */
    plasma->nb = 128;
    plasma->ib = 32;
    plasma->nbnbsize = 16384;
    plasma->ibnbsize = 4096;
    plasma->info = 0;

    plasma->errors_enabled     = PLASMA_FALSE;
    plasma->warnings_enabled   = PLASMA_FALSE;
    plasma->autotuning_enabled = PLASMA_TRUE;

    return plasma;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
// Insert a (context, thread_id) tuple in the context map
int plasma_context_insert(plasma_context_t *context, pthread_t thread_id)
{
    int i;

    // acquire the access lock
    pthread_mutex_lock(&context_map_lock);
    // For each entry
    for (i = 0; i < CONTEXTS_MAX; i++) {
        // If not occupied
        if (context_map[i].context == NULL) {
            // Insert new context, release lock, return success
            context_map[i].context = context;
            context_map[i].thread_id = thread_id;
            pthread_mutex_unlock(&context_map_lock);
            return PLASMA_SUCCESS;
        }
    }
    // No empty entry found - release lock, return error
    pthread_mutex_unlock(&context_map_lock);
    plasma_fatal_error("plasma_context_insert", "too many threads");
    return PLASMA_ERR_INTERNAL_LIMIT;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
// Remove a (context, thread_id) tuple from the context map
int plasma_context_remove(plasma_context_t *context, pthread_t thread_id)
{
    int i;

    // acquire the access lock
    pthread_mutex_lock(&context_map_lock);
    // For each entry
    for (i = 0; i < CONTEXTS_MAX; i++) {
        // If id matches
        if (pthread_equal(context_map[i].thread_id, thread_id)) {
            if (context_map[i].context == context) {
                // Free the context, mark entry as empty, release lock, return success
                free(context_map[i].context);
                context_map[i].context = NULL;
                pthread_mutex_unlock(&context_map_lock);
                return PLASMA_SUCCESS;
            }
            else {
                pthread_mutex_unlock(&context_map_lock);
                plasma_fatal_error("plasma_context_remove", "context does not match thread");
                return PLASMA_ERR_UNEXPECTED;
            }
        }
    }
    // No matching id found - release lock, return error
    pthread_mutex_unlock(&context_map_lock);
    plasma_fatal_error("plasma_context_remove", "thread not found");
    return PLASMA_ERR_NOT_FOUND;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
// Return context for a thread
plasma_context_t *plasma_context_self()
{
    int i;

    // For each entry
    for (i = 0; i < CONTEXTS_MAX; i++) {
        // If id matches
        if (pthread_equal(context_map[i].thread_id, pthread_self())) {
            return context_map[i].context;
        }
    }
    return NULL;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Enable - Enable PLASMA feature.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// lever    PLASMA_enum (IN)
//          Feature to be enabled:
//          = PLASMA_WARNINGS:   printing of warning messages,
//          = PLASMA_ERRORS:     printing of error messages,
//          = PLASMA_AUTOTUNING: autotuning for tile size and inner block size.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Enable(PLASMA_enum lever)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_Enable", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }

    switch (lever)
    {
        case PLASMA_WARNINGS:
            plasma->warnings_enabled = PLASMA_TRUE;
            break;
        case PLASMA_ERRORS:
            plasma->errors_enabled = PLASMA_TRUE;
            break;
        case PLASMA_AUTOTUNING:
            plasma->autotuning_enabled = PLASMA_TRUE;
            break;
        default:
            plasma_error("PLASMA_Enable", "illegal parameter value");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Disable - Disable PLASMA feature.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// lever    PLASMA_enum (IN)
//          Feature to be disabled:
//          = PLASMA_WARNINGS:   printing of warning messages,
//          = PLASMA_ERRORS:     printing of error messages,
//          = PLASMA_AUTOTUNING: autotuning for tile size and inner block size.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Disable(PLASMA_enum lever)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_Disable", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    switch (lever)
    {
        case PLASMA_WARNINGS:
            plasma->warnings_enabled = PLASMA_FALSE;
            break;
        case PLASMA_ERRORS:
            plasma->errors_enabled = PLASMA_FALSE;
            break;
        case PLASMA_AUTOTUNING:
            plasma->autotuning_enabled = PLASMA_FALSE;
            break;
        default:
            plasma_error("PLASMA_Disable", "illegal parameter value");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Set - Set PLASMA parameter

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// param    PLASMA_enum (IN)
//          PLASMA parameter:
//          = PLASMA_TILE_SIZE:        size matrix tile,
//          = PLASMA_INNER_BLOCK_SIZE: size of tile inner block.
//
// value    int (IN)
//          Value of the parameter.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Set(PLASMA_enum param, int value)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_error("PLASMA_Set", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    switch (param) {
        case PLASMA_TILE_SIZE:
            if (value <= 0) {
                plasma_error("PLASMA_Set", "negative tile size");
                return PLASMA_ERR_ILLEGAL_VALUE;
            }
            plasma->nb = value;
            /* Calculate A, B tile size and round up to cache line size */
            /* round up for the smallest type (float) - will hold for all */
            plasma->nbnbsize = plasma->nb * plasma->nb; // * sizeof(float);
//          plasma->nbnbsize = roundup(plasma->nbnbsize, CACHE_LINE_SIZE);
//          plasma->nbnbsize /= sizeof(float);
            break;
        case PLASMA_INNER_BLOCK_SIZE:
            if (value <= 0) {
                plasma_error("PLASMA_Set", "negative inner block size");
                return PLASMA_ERR_ILLEGAL_VALUE;
            }
            if (value > plasma->nb) {
                plasma_error("PLASMA_Set", "inner block larger than tile");
                return PLASMA_ERR_ILLEGAL_VALUE;
            }
            if (plasma->nb % value != 0) {
                plasma_error("PLASMA_Set", "inner block does not divide tile");
                return PLASMA_ERR_ILLEGAL_VALUE;
            }
            plasma->ib = value;
            /* Calculate T, L tile size and round up to cache line size */
            /* round up for the smallest type (float) - will hold for all */
            plasma->ibnbsize = plasma->ib * plasma->nb; // * sizeof(float);
//          plasma->ibnbsize = roundup(plasma->ibnbsize, CACHE_LINE_SIZE);
//          plasma->ibnbsize /= sizeof(float);
            break;
        default:
            plasma_error("PLASMA_Set", "unknown parameter");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Get - Get value of PLASMA parameter

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// param    PLASMA_enum (IN)
//          PLASMA parameter:
//          = PLASMA_TILE_SIZE:        size matrix tile,
//          = PLASMA_INNER_BLOCK_SIZE: size of tile inner block.
//
// value    int* (OUT)
//          Value of the parameter.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Get(PLASMA_enum param, int *value)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_Get", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    switch (param) {
        case PLASMA_TILE_SIZE:
            *value = plasma->nb;
            return PLASMA_SUCCESS;
        case PLASMA_INNER_BLOCK_SIZE:
            *value = plasma->ib;
            return PLASMA_SUCCESS;
        default:
            plasma_error("PLASMA_Get", "unknown parameter");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
}
