/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.1.0)                          ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA facilities of interest to both PLASMA core developer and also of interest to
//  PLASMA community contributor.
#ifndef _PLASMA_COMMON_H_
#define _PLASMA_COMMON_H_

#include "global.h"
#include "context.h"
#include "control.h"
#include "core_blas.h"
#include "allocate.h"
#include "auxiliary.h"
#include "descriptor.h"
#include "tile.h"

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Global shortcuts
#define PLASMA_RANK plasma_rank(plasma)
#define PLASMA_SIZE plasma->world_size
#define PLASMA_NB plasma->nb
#define PLASMA_IB plasma->ib
#define PLASMA_NBNBSIZE plasma->nbnbsize
#define PLASMA_IBNBSIZE plasma->ibnbsize
#define PLASMA_INFO plasma->info

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Global utilities
#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif
#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif
#define roundup(a, b) (b <= 0) ? (a) : (((a) + (b)-1) & ~((b)-1))

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Static scheduler
#define ss_init(m, n, init_val) \
{ \
    if (PLASMA_RANK == 0) { \
        plasma->ss_progress = (volatile int *)plasma_shared_alloc(plasma, (m)*(n), PlasmaInteger); \
        plasma_memset_int((int*)plasma->ss_progress, (m)*(n), (init_val)); \
    } \
    plasma->ss_ld = (m); \
    plasma->ss_abort = 0; \
    plasma_barrier(plasma); \
}

#define ss_abort()   plasma->ss_abort = 1;
#define ss_aborted() plasma->ss_abort

#define ss_cond_set(m, n, val) \
{ \
    plasma->ss_progress[(m)+plasma->ss_ld*(n)] = (val); \
}

#define ss_cond_wait(m, n, val) \
{ \
    while (!plasma->ss_abort && \
            plasma->ss_progress[(m)+plasma->ss_ld*(n)] != (val)); \
}

#define ss_finalize() \
{ \
    plasma_barrier(plasma); \
    if (PLASMA_RANK == 0) \
        plasma_shared_free(plasma, (void*)plasma->ss_progress); \
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Global array of LAPACK constants
extern char *plasma_lapack_constants[];

#ifdef __cplusplus
extern "C" {
#endif

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Parallel numerical functions prototypes
void plasma_psgelqf(plasma_context_t *plasma);
void plasma_psgeqrf(plasma_context_t *plasma);
void plasma_psgetrf(plasma_context_t *plasma);
void plasma_psorglq(plasma_context_t *plasma);
void plasma_psorgqr(plasma_context_t *plasma);
void plasma_psormlq(plasma_context_t *plasma);
void plasma_psormqr(plasma_context_t *plasma);
void plasma_pspotrf(plasma_context_t *plasma);
void plasma_pstrsm(plasma_context_t *plasma);
void plasma_pstrsmpl(plasma_context_t *plasma);
void plasma_psgemm(plasma_context_t *plasma);

void plasma_pdgelqf(plasma_context_t *plasma);
void plasma_pdgeqrf(plasma_context_t *plasma);
void plasma_pdgetrf(plasma_context_t *plasma);
void plasma_pdorglq(plasma_context_t *plasma);
void plasma_pdorgqr(plasma_context_t *plasma);
void plasma_pdormlq(plasma_context_t *plasma);
void plasma_pdormqr(plasma_context_t *plasma);
void plasma_pdpotrf(plasma_context_t *plasma);
void plasma_pdtrsm(plasma_context_t *plasma);
void plasma_pdtrsmpl(plasma_context_t *plasma);
void plasma_pdgemm(plasma_context_t *plasma);

void plasma_pcgelqf(plasma_context_t *plasma);
void plasma_pcgeqrf(plasma_context_t *plasma);
void plasma_pcgetrf(plasma_context_t *plasma);
void plasma_pcunglq(plasma_context_t *plasma);
void plasma_pcungqr(plasma_context_t *plasma);
void plasma_pcunmlq(plasma_context_t *plasma);
void plasma_pcunmqr(plasma_context_t *plasma);
void plasma_pcpotrf(plasma_context_t *plasma);
void plasma_pctrsm(plasma_context_t *plasma);
void plasma_pctrsmpl(plasma_context_t *plasma);
void plasma_pcgemm(plasma_context_t *plasma);

void plasma_pzgelqf(plasma_context_t *plasma);
void plasma_pzgeqrf(plasma_context_t *plasma);
void plasma_pzgetrf(plasma_context_t *plasma);
void plasma_pzunglq(plasma_context_t *plasma);
void plasma_pzungqr(plasma_context_t *plasma);
void plasma_pzunmlq(plasma_context_t *plasma);
void plasma_pzunmqr(plasma_context_t *plasma);
void plasma_pzpotrf(plasma_context_t *plasma);
void plasma_pztrsm(plasma_context_t *plasma);
void plasma_pztrsmpl(plasma_context_t *plasma);
void plasma_pzgemm(plasma_context_t *plasma);


/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Parallel auxiliary functions prototypes


void plasma_pslange(plasma_context_t *plasma);
void plasma_pslacpy(plasma_context_t *plasma);
void plasma_psaxpy(plasma_context_t *plasma);

void plasma_pdlange(plasma_context_t *plasma);
void plasma_pdlacpy(plasma_context_t *plasma);
void plasma_pdaxpy(plasma_context_t *plasma);

void plasma_pclange(plasma_context_t *plasma);
void plasma_pclacpy(plasma_context_t *plasma);
void plasma_pcaxpy(plasma_context_t *plasma);

void plasma_pzlange(plasma_context_t *plasma);
void plasma_pzlacpy(plasma_context_t *plasma);
void plasma_pzaxpy(plasma_context_t *plasma);

void plasma_pdlag2s(plasma_context_t *plasma);
void plasma_pzlag2c(plasma_context_t *plasma);
void plasma_pslag2d(plasma_context_t *plasma);
void plasma_pclag2z(plasma_context_t *plasma);

#ifdef __cplusplus
}
#endif

#endif
