/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routines (version 2.1.0)                      ///
 * ///                    Author: Jakub Kurzak                                               ///
 * ///                    Release Date: November, 15th 2009                                  ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_cgelqf - Computes the tile LQ factorization of a complex M-by-N matrix A: A = L * Q.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A. N >= 0.
//
// A        PLASMA_Complex32_t* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit, the elements on and below the diagonal of the array contain the m-by-min(M,N)
//          lower trapezoidal matrix L (L is lower triangular if M <= N); the elements above the
//          diagonal represent the unitary matrix Q as a product of elementary reflectors, stored
//          by tiles.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// T        PLASMA_Complex32_t* (OUT)
//          On exit, auxiliary factorization data, required by PLASMA_cgelqs to solve the system
//          of equations.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_cgelqf(int M, int N, PLASMA_Complex32_t *A, int LDA, PLASMA_Complex32_t *T)
{
    int NB, MT, NT;
    int status;
    PLASMA_Complex32_t *Abdl;
    PLASMA_Complex32_t *Tbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_cgelqf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (M < 0) {
        plasma_error("PLASMA_cgelqf", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_cgelqf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_cgelqf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_CGELS, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_cgelqf", "plasma_tune() failed");
        return status;
    }

    /* Set MT & NT */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex32_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaComplexFloat);
    Tbdl = (PLASMA_Complex32_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexFloat);
    if (Abdl == NULL || Tbdl == NULL) {
        plasma_error("PLASMA_cgelqf", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Tbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexFloat,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        M, N, 0, 0, M, N);

    PLASMA_desc descT = plasma_desc_init(
        Tbdl, PlasmaComplexFloat,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        M, N, 0, 0, M, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex32_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    /* Call the native interface */
    status = PLASMA_cgelqf_Tile(&descA, &descT);

    if (status == PLASMA_SUCCESS) {
        /* Return T to the user */
        plasma_memcpy(T, Tbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexFloat);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex32_t*, A,
            int, LDA);
    }
    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Tbdl);
    return status;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_cgelqf_Tile - Computes the tile LQ factorization of a complex M-by-N matrix A: A = L * Q.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        PLASMA_Complex32_t* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit, the elements on and below the diagonal of the array contain the m-by-min(M,N)
//          lower trapezoidal matrix L (L is lower triangular if M <= N); the elements above the
//          diagonal represent the unitary matrix Q as a product of elementary reflectors, stored
//          by tiles.
//
// T        PLASMA_Complex32_t* (OUT)
//          On exit, auxiliary factorization data, required by PLASMA_cgelqs to solve the system
//          of equations.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_cgelqf_Tile(PLASMA_desc *A, PLASMA_desc *T)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descT = *T;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_cgelqf_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_cgelqf_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descT) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_cgelqf_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (descA.nb != descA.mb) {
        plasma_error("PLASMA_cgelqf_Tile", "only square tiles supported");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Quick return */
/*
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_2(plasma_pcgelqf,
        PLASMA_desc, descA,
        PLASMA_desc, descT);

    return PLASMA_SUCCESS;
}
