      SUBROUTINE CORE_DGEQRT( M, N, IB, A, LDA, T, LDT,
     $                       TAU, WORK, INFO )

*********************************************************************
*     PLASMA core_blas routine (version 2.1.0)                      *
*     Author: Hatem Ltaief                                          *
*     Release Date: November, 15th 2009                             *
*     PLASMA is a software package provided by Univ. of Tennessee,  *
*     Univ. of California Berkeley and Univ. of Colorado Denver.    *
*********************************************************************
*
*     .. Scalar Arguments ..
      INTEGER            M, N, IB, LDA, LDT, INFO
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION            A( LDA, * ), T( LDT, * )
      DOUBLE PRECISION            TAU( * ), WORK( * )
*     ..
*
*  Purpose
*  =======
*
*  CORE_DGEQRT computes a QR factorization of a real M-by-N tile A:
*  A = Q * R.
*
*  Arguments
*  =========
*
*  M       (input) INTEGER
*          The number of rows of the tile A.  M >= 0.
*
*  N       (input) INTEGER
*          The number of columns of the tile A.  N >= 0.
*
*  IB      (input) INTEGER
*          The inner-blocking size.  IB >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the M-by-N tile A.
*          On exit, the elements on and above the diagonal of the array
*          contain the min(M,N)-by-N upper trapezoidal tile R (R is
*          upper triangular if M >= N); the elements below the diagonal,
*          with the array TAU, represent the orthogonal tile Q as a
*          product of elementary reflectors (see Further Details).
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,M).
*
*  T       (output) DOUBLE PRECISION array, dimension (LDT,N)
*          The IB-by-N triangular factor T of the block reflector.
*          T is upper triangular by block (economic storage);
*          The rest of the array is not referenced.
*
*  LDT     (input) INTEGER
*          The leading dimension of the array T. LDT >= IB.
*
*  TAU     (output) DOUBLE PRECISION array, dimension (min(M,N))
*          The scalar factors of the elementary reflectors (see Further
*          Details).
*
*  WORK    (workspace) DOUBLE PRECISION array, dimension (N)
*
*  INFO    (output) INTEGER
*          = 0: successful exit
*          < 0: if INFO = -i, the i-th argument had an illegal value
*
*  Further Details
*  ===============
*
*  The tile Q is represented as a product of elementary reflectors
*
*     Q = H(1) H(2) . . . H(k), where k = min(M,N).
*
*  Each H(i) has the form
*
*     H(i) = I - tau * v * v'
*
*  where tau is a real scalar, and v is a real vector with
*  v(1:i-1) = 0 and v(i) = 1; v(i+1:m) is stored on exit in A(i+1:m,i),
*  and tau in TAU(i).
*
*  =====================================================================
*
*     ..
*     .. Local Scalars ..
      INTEGER            I, K, SB, IINFO
*     ..
*     .. External Subroutines ..
      EXTERNAL           XERBLA, DGEQR2, DLARFT, CORE_DORMQR
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN
*     ..
*     Test the input arguments.
      INFO = 0
      IF( M.LT.0 ) THEN
         INFO = -1
      ELSE IF( N.LT.0 ) THEN
         INFO = -2
      ELSE IF( IB.LT.0 ) THEN
         INFO = -3
      ELSE IF( LDA.LT.MAX( 1, M ) ) THEN
         INFO = -5
      ELSE IF( LDT.LT.MAX( 1, IB ) ) THEN
         INFO = -7
      END IF
      IF( INFO.NE.0 ) THEN
         CALL XERBLA( 'CORE_DGEQRT', -INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( M.EQ.0 .OR. N.EQ.0 .OR. IB.EQ.0 )
     $   RETURN
*
      K = MIN( M, N )
*
      DO 10 I = 1, K, IB
         SB = MIN( IB, K-I+1 )
         CALL DGEQR2( M-I+1, SB, A( I, I ), LDA, TAU( I ), WORK, IINFO )
*
         CALL DLARFT( 'Forward', 'Columnwise', M-I+1, SB,
     $               A( I, I ), LDA, TAU( I ),
     $               T( 1, I ), LDT )
*
         CALL CORE_DORMQR( 'Left', 'Transpose',
     $                    M-I+1, N-I+1-SB, SB, SB,
     $                    A( I, I ), LDA,
     $                    T( 1, I ), LDT,     
     $                    A( I, I+SB ), LDA,
     $                    WORK, N-I+1-SB, IINFO )
 10   CONTINUE
*
      RETURN
*
*     End of CORE_DGEQRT.
*
      END
