      SUBROUTINE CORE_DGELQT( M, N, IB, A, LDA, T, LDT,
     $                       TAU, WORK, INFO )

*********************************************************************
*     PLASMA core_blas routine (version 2.1.0)                      *
*     Author: Hatem Ltaief                                          *
*     Release Date: November, 15th 2009                             *
*     PLASMA is a software package provided by Univ. of Tennessee,  *
*     Univ. of California Berkeley and Univ. of Colorado Denver.    *
*********************************************************************
*
*     .. Scalar Arguments ..
      INTEGER            M, N, IB, LDA, LDT, INFO
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION            A( LDA, * ), T( LDT, * )
      DOUBLE PRECISION            TAU( * ), WORK( * )
*     ..
*
*  Purpose
*  =======
*
*  CORE_DGELQT computes a LQ factorization of a real M-by-N tile A:
*  A = L * Q.
*
*  Arguments
*  =========
*
*  M       (input) INTEGER
*          The number of rows of the tile A.  M >= 0.
*
*  N       (input) INTEGER
*          The number of columns of the tile A.  N >= 0.
*
*  IB      (input) INTEGER
*          The inner-blocking size.  IB >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the M-by-N tile A.
*          On exit, the elements on and below the diagonal of the array
*          contain the M-by-min(M,N) lower trapezoidal tile L (L is
*          lower triangular if M <= N); the elements above the diagonal,
*          with the array TAU, represent the orthogonal tile Q as a
*          product of elementary reflectors (see Further Details).
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,M).
*
*  T       (output) DOUBLE PRECISION array, dimension (LDT,N)
*          The IB-by-N triangular factor T of the block reflector.
*          T is upper triangular by block (economic storage);
*          The rest of the array is not referenced.
*
*  LDT     (input) INTEGER
*          The leading dimension of the array T. LDT >= IB.
*
*  TAU     (output) DOUBLE PRECISION array, dimension (min(M,N))
*          The scalar factors of the elementary reflectors (see Further
*          Details).
*
*  WORK    (workspace) DOUBLE PRECISION array, dimension (M)
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*
*  Further Details
*  ===============
*
*  The tile Q is represented as a product of elementary reflectors
*
*     Q = H(k)' . . . H(2)' H(1)', where k = min(M,N).
*
*  Each H(i) has the form
*
*     H(i) = I - tau * v * v'
*
*  where tau is a real scalar, and v is a real vector with
*  v(1:i-1) = 0 and v(i) = 1; conjg(v(i+1:n)) is stored on exit in
*  A(i,i+1:n), and tau in TAU(i).
*
*  =====================================================================
*
*     .. Local Scalars ..
      INTEGER            I, K, SB, IINFO
*     ..
*     .. External Subroutines ..
      EXTERNAL           XERBLA, DGELQ2, DLARFT, CORE_DORMLQ
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN
*     ..
*     Test the input arguments
      INFO = 0
      IF( M.LT.0 ) THEN
         INFO = -1
      ELSE IF( N.LT.0 ) THEN
         INFO = -2
      ELSE IF( IB.LT.0 ) THEN
         INFO = -3
      ELSE IF( LDA.LT.MAX( 1, M ) ) THEN
         INFO = -5
      ELSE IF( LDT.LT.MAX( 1, IB ) ) THEN
         INFO = -7
      END IF
      IF( INFO.NE.0 ) THEN
         CALL XERBLA( 'CORE_DGELQT', -INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( M.EQ.0 .OR. N.EQ.0 .OR. IB.EQ.0 )
     $   RETURN
*
      K = MIN( M, N )
*
      DO 10 I = 1, K, IB
         SB = MIN( IB, K-I+1 )
         CALL DGELQ2( SB, N-I+1, A( I, I ), LDA, TAU( I ), WORK, IINFO )
*
         CALL DLARFT( 'Forward', 'Rowwise', N-I+1, SB,
     $                A( I, I ),LDA, TAU( I ),
     $                T( 1, I ), LDT )
*
         IF ( M.GT.( I+IB-1 ) ) THEN   
             CALL CORE_DORMLQ( 'Right', 'Transpose',
     $                        M-I+1-SB, N-I+1, SB, SB,
     $                        A( I, I ), LDA,
     $                        T( 1, I ), LDT,
     $                        A( I+SB, I ), LDA,
     $                        WORK, M-I+1-SB, IINFO )
         ENDIF
 10   CONTINUE
*
      RETURN
*
*     End of CORE_DGELQT.
*
      END
