      SUBROUTINE CORE_CTSTRF( M, N, IB, NB, U, LDU, A, LDA,
     $                       L, LDL, IPIV, WORK, LDWORK, INFO )

*********************************************************************
*     PLASMA core_blas routine (version 2.1.0)                      *
*     Author: Hatem Ltaief                                          *
*     Release Date: November, 15th 2009                             *
*     PLASMA is a software package provided by Univ. of Tennessee,  *
*     Univ. of California Berkeley and Univ. of Colorado Denver.    *
*********************************************************************
*
*     .. Scalar Arguments ..
      INTEGER            M, N, IB, NB, LDU, LDA, LDL, LDWORK, INFO
*     ..
*     .. Array Arguments ..
      COMPLEX            U( LDU, * ), A( LDA, * )
      COMPLEX            L( LDL, * ), WORK( LDWORK, * )
      INTEGER            IPIV( * )
*     ..
*
*  Purpose
*  =======
*
*  CORE_CTSTRF computes an LU factorization of a complex matrix formed
*  by an upper triangular NB-by-N tile U on top of a M-by-N tile A
*  using partial pivoting with row interchanges.
*
*  This is the right-looking Level 2.5 BLAS version of the algorithm.
*
*  Arguments
*  =========
*
*  M       (input) INTEGER
*          The number of rows of the tile A.  M >= 0.
*
*  N       (input) INTEGER
*          The number of columns of the tile A.  N >= 0.
*
*  IB      (input) INTEGER
*          The inner-blocking size.  IB >= 0.
*
*  U       (input/output) COMPLEX array, dimension (LDU,N)
*          On entry, the NB-by-N upper triangular tile.
*          On exit, the new factor U from the factorization
*
*  LDU     (input) INTEGER
*          The leading dimension of the array U.  LDU >= max(1,NB).
*
*  A       (input/output) COMPLEX array, dimension (LDA,N)
*          On entry, the M-by-N tile to be factored.
*          On exit, the factor L from the factorization
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,M).
*
*  L       (input/output) COMPLEX array, dimension (LDL,N)
*          On entry, the NB-by-N lower triangular tile.
*          On exit, the interchanged rows formthe tile A in case of pivoting.
*
*  LDL     (input) INTEGER
*          The leading dimension of the array L.  LDL >= max(1,NB).
*
*  IPIV    (output) INTEGER array, dimension (min(M,N))
*          The pivot indices; for 1 <= i <= min(M,N), row i of the
*          tile U was interchanged with row IPIV(i) of the tile A.
*
*  WORK    (workspace) COMPLEX array, dimension (LDWORK,IB)
*
*  LDWORK  (input) INTEGER
*          The dimension of the array WORK.
*
*  INFO    (output) INTEGER
*          = 0: successful exit
*          < 0: if INFO = -k, the k-th argument had an illegal value
*          > 0: if INFO = k, U(k,k) is exactly zero. The factorization
*               has been completed, but the factor U is exactly
*               singular, and division by zero will occur if it is used
*               to solve a system of equations.
*
*  =====================================================================
*     ..
*     Internal variables ..
      INTEGER            I, II, J, SB, IM, IP, IDAMAX, IINFO
      EXTERNAL           IDAMAX
*     ..
*     .. Parameters ..
      COMPLEX            MCONE, CZERO
      PARAMETER          ( MCONE = ( -1.0E+0, 0.0E+0 ) )
      PARAMETER          ( CZERO = ( 0.0E+0, 0.0E+0 ) )
*     ..
*     Test the input parameters.
      INFO = 0
      IF( M.LT.0 ) THEN
         INFO = -1
      ELSE IF( N.LT.0 ) THEN
         INFO = -2
      ELSE IF( IB.LT.0 ) THEN
         INFO = -3
      ELSE IF( LDU.LT.MAX( 1, M ) ) THEN
         INFO = -6
      ELSE IF( LDA.LT.MAX( 1, M ) ) THEN
         INFO = -8
      ELSE IF( LDL.LT.MAX( 1, IB ) ) THEN
         INFO = -10
      END IF
      IF( INFO.NE.0 ) THEN
         CALL XERBLA( 'CORE_CTSTRF', -INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( M.EQ.0 .OR. N.EQ.0 .OR. IB.EQ.0 )
     $   RETURN
*
      IP = 1
*
      DO 20 II = 1, N, IB
         SB = MIN( N-II+1, IB )
         DO 10 I = 1, SB
*
            IM = IDAMAX( M, A( 1, II+I-1 ), 1 )
            IPIV( IP ) = II+I-1
*
            IF( ABS( A( IM, II+I-1 ) ) 
     $          .GT. ABS( U( II+I-1, II+I-1 ) ) ) THEN
*
*              Swap behind.
*
               CALL CSWAP( I-1, L( I, II ), LDL,
     $                    WORK( IM, 1 ), LDWORK )
*
*              Swap ahead.
*
               CALL CSWAP( SB-I+1, U( II+I-1, II+I-1 ), LDU,
     $                    A( IM, II+I-1 ), LDA )
*
*              Set IPIV.
*
               IPIV( IP ) = NB + IM
*
               DO 50 J = 1, I-1
                  A( IM, II+J-1 ) = CZERO
 50            CONTINUE
*
            END IF
            IF( INFO.EQ.0 .AND. ABS(A(IM,II+I-1)).EQ.CZERO 
     $          .AND. ABS(U(II+I-1, II+I-1)).EQ.CZERO ) THEN
               INFO = II+I-1
            END IF
*
            CALL CSCAL( M, 1/U( II+I-1, II+I-1 ), A( 1, II+I-1 ), 1 )
            CALL CCOPY( M, A( 1, II+I-1 ), 1, WORK( 1, I ), 1 )
            CALL CGERU( M, SB-I, MCONE, A( 1, II+I-1 ), 1,
     $                U( II+I-1, II+I ), LDU,
     $                A( 1, II+I ), LDA )
*
            IP = IP+1
 10      CONTINUE
*
*        Apply the subpanel to the rest of the panel.
*
         IF( II+I-1.LE. N ) THEN
            DO 80 J = II, II+SB-1
               IF( IPIV( J ).LE.NB ) THEN
                  IPIV( J ) = IPIV( J )-II+1
               ENDIF
 80         CONTINUE
*
            CALL CORE_CSSSSM( NB, M, N-( II+SB-1 ), SB, SB,
     $                       U( II, II+SB ), LDU,
     $                       A( 1, II+SB ), LDA,
     $                       L( 1, II ), LDL,
     $                       WORK, LDWORK, IPIV( II ),
     $                       IINFO )
*
            DO 70 J = II, II+SB-1
               IF( IPIV( J ).LE.NB ) THEN
                  IPIV( J ) = IPIV( J )+II-1
               ENDIF
 70         CONTINUE
         END IF
 20   CONTINUE
*
      RETURN
*
*     End of CORE_CTSTRF.
*
      END
