#! /usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import os
import string

# The master file needs to be either COMPLEX or COMPLEX*16. If it's not the
# case many things might break. Among others, these things will break:
#  - data type for SLAMCH/DLAMCH (in COMPLEX and COMPLEX*16 they have a unique
#    data type: either REAL or DOUBLE PRECISION),
#  - data type for SFMIN (same problem as with SLAMCH/DLAMCH),
#  - use of CONJG (it's not present in REAL and DOUBLE PRECISION - the script
#    will not add it automatically),
#  - call to CLACGV (it's missing in REAL and DOUBLE PRECISION versions -
#    the script will not add it out of thin air)

G_F_Variables = """
REAL;;;DOUBLE_PRECISION;;;COMPLEX;;;COMPLEX_16
"""

G_C_Variables = """
REAL;;;DOUBLE_PRECISION;;;COMPLEX;;;COMPLEX_16
"""

G_H_Variables = """
REAL;;;DOUBLE_PRECISION;;;COMPLEX;;;COMPLEX_16
"""

G_F_translations = """
real;;;real;;;complex;;;complex
REAL;;;DOUBLE PRECISION;;;REAL;;;DOUBLE PRECISION
REAL;;;DOUBLE PRECISION;;;COMPLEX;;;COMPLEX*16
SGER;;;DGER;;;CGERU;;;ZGERU
SGER;;;DGER;;;CGERC;;;ZGERC
SDOT;;;DDOT;;;CDOT;;;ZDOT
SSCAL;;;DSCAL;;;CSCAL;;;ZSCAL
SAXPY;;;DAXPY;;;CAXPY;;;ZAXPY
STRSM;;;DTRSM;;;CTRSM;;;ZTRSM
STRMV;;;DTRMV;;;CTRMV;;;ZTRMV
STRMM;;;DTRMM;;;CTRMM;;;ZTRMM
SGEMV;;;DGEMV;;;CGEMV;;;ZGEMV
SGEMM;;;DGEMM;;;CGEMM;;;ZGEMM
SLANRV;;;DLANRV;;;CLANRV;;;ZLANRV
SSYRK;;;DSYRK;;;CHERK;;;ZHERK
SPOTF2;;;DPOTF2;;;CPOTF2;;;ZPOTF2
SGEQR2;;;DGEQR2;;;CGEQR2;;;ZGEQR2
SPOSV;;;DPOSV;;;CPOSV;;;ZPOSV
SGESV;;;DGESV;;;CGESV;;;ZGESV
SGETF2;;;DGETF2;;;CGETF2;;;ZGETF2
SGELQ2;;;DGELQ2;;;CGELQ2;;;ZGELQ2
SLARFG;;;DLARFG;;;CLARFG;;;ZLARFG
SLARFB;;;DLARFB;;;CLARFB;;;ZLARFB
SLARFT;;;DLARFT;;;CLARFT;;;ZLARFT
SLARUV;;;DLARUV;;;CLARUV;;;ZLARUV
SLACGV;;;DLACGV;;;CLACGV;;;ZLACGV
SORM2R;;;DORM2R;;;CORM2R;;;ZORM2R
SORML2;;;DORML2;;;CORML2;;;ZORML2
SISNAN;;;DISNAN;;;CISNAN;;;ZISNAN
SLAISNAN;;;DLAISNAN;;;CLAISNAN;;;ZLAISNAN
SLAPY2;;;DLAPY2;;;CLAPY2;;;ZLAPY2
SLAMCH;;;DLAMCH;;;SLAMCH;;;DLAMCH
SNRM2;;;DNRM2;;;SNRM2;;;DNRM2
SCOPY;;;DCOPY;;;CCOPY;;;ZCOPY
SLACPY;;;DLACPY;;;CLACPY;;;ZLACPY
SSWAP;;;DSWAP;;;CSWAP;;;ZSWAP
SLASWP;;;DLASWP;;;CLASWP;;;ZLASWP
SGETRF;;;DGETRF;;;CGETRF;;;ZGETRF
SGETRS;;;DGETRS;;;CGETRS;;;ZGETRS
SLANGE;;;DLANGE;;;SLANGE;;;DLANGE
SLASSQ;;;DLASSQ;;;CLASSQ;;;ZLASSQ
SLARNV;;;DLARNV;;;CLARNV;;;ZLARNV
SLARNV;;;DLARNV;;;SLARNV;;;DLARNV
SLAGSY;;;DLAGSY;;;CLAGHE;;;ZLAGHE
CORE_S;;;CORE_D;;;CORE_C;;;CORE_Z
ORMQR;;;ORMQR;;;UNMQR;;;UNMQR
ORMLQ;;;ORMLQ;;;UNMLQ;;;UNMLQ
PLASMA_S;;;PLASMA_D;;;PLASMA_C;;;PLASMA_Z
PLASMA_ALLOC_WORKSPACE_S;;;PLASMA_ALLOC_WORKSPACE_D;;;PLASMA_ALLOC_WORKSPACE_C;;;PLASMA_ALLOC_WORKSPACE_Z
TRANS, 'T';;;TRANS, 'T';;;TRANS, 'C';;;TRANS, 'C'
TRANST = 'T';;;TRANST = 'T';;;TRANST = 'C';;;TRANST = 'C'
( SONE = 1.0E+0 );;;( DONE = 1.0D+0 );;;( CONE = ( 1.0E+0, 0.0E+0 ) );;;( ZONE = ( 1.0D+0, 0.0D+0 ) )
( MSONE = -1.0E+0 );;;( MDONE = -1.0D+0 );;;( MCONE = ( -1.0E+0, 0.0E+0 ) );;;( MZONE = ( -1.0D+0, 0.0D+0 ) )
( SZERO = 0.0E+0 );;;( DZERO = 0.0D+0 );;;( CZERO = ( 0.0E+0, 0.0E+0 ) );;;( ZZERO = ( 0.0D+0, 0.0D+0 ) )
SONE;;;DONE;;;CONE;;;ZONE
MSONE;;;MDONE;;;MCONE;;;MZONE
SZERO;;;DZERO;;;CZERO;;;ZZERO
orthogonal;;;orthogonal;;;unitary;;;unitary
'Transpose';;;'Transpose';;;'ConjTranspose';;;'ConjTranspose'
"""

G_F_deletions = """
CONJG,;;;CONJG,;;; ;;;
, CONJG;;;, CONJG;;; ;;;
CONJG;;;CONJG;;; ;;;
"""

G_C_translations = """
float;;;double;;;PLASMA_Complex32_t;;;PLASMA_Complex64_t
float;;;double;;;float _Complex;;;double _Complex
float;;;double;;;float;;;double
PLASMA_sor;;;PLASMA_dor;;;PLASMA_cun;;;PLASMA_zun
PLASMA_S;;;PLASMA_D;;;PLASMA_C;;;PLASMA_Z
PLASMA_s;;;PLASMA_d;;;PLASMA_c;;;PLASMA_z
PLASMA_xs;;;PLASMA_xd;;;PLASMA_xc;;;PLASMA_xz
plasma_s;;;plasma_d;;;plasma_c;;;plasma_z
plasma_ps;;;plasma_pd;;;plasma_pc;;;plasma_pz
PlasmaRealFloat;;;PlasmaRealDouble;;;PlasmaComplexFloat;;;PlasmaComplexDouble
PlasmaTrans;;;PlasmaTrans;;;PlasmaConjTrans;;;PlasmaConjTrans
CblasTrans;;;CblasTrans;;;CblasConjTrans;;;CblasConjTrans
cblas_s;;;cblas_d;;;cblas_c;;;cblas_z
CORE_s;;;CORE_d;;;CORE_c;;;CORE_z
core_s;;;core_d;;;core_c;;;core_z
slamch;;;dlamch;;;slamch;;;dlamch
slange;;;dlange;;;clange;;;zlange
slansy;;;dlansy;;;clansy;;;zlansy
slacpy;;;dlacpy;;;clacpy;;;zlacpy
spotrf;;;dpotrf;;;cpotrf;;;zpotrf
slarnv;;;dlarnv;;;clarnv;;;zlarnv
slarnv;;;dlarnv;;;slarnv;;;dlarnv
slagsy;;;dlagsy;;;claghe;;;zlaghe
ormqr;;;ormqr;;;unmqr;;;unmqr
ormlq;;;ormlq;;;unmlq;;;unmlq
orgqr;;;orgqr;;;ungqr;;;ungqr
orglq;;;orglq;;;unglq;;;unglq
SORMQR;;;DORMQR;;;CUNMQR;;;ZUNMQR
SORMLQ;;;DORMLQ;;;CUNMLQ;;;ZUNMLQ
SORGQR;;;DORGQR;;;CUNGQR;;;ZUNGQR
SORGLQ;;;DORGLQ;;;CUNGLQ;;;ZUNGLQ
ORGQR;;;ORGQR;;;UNGQR;;;UNGQR
ORGLQ;;;ORGLQ;;;UNGLQ;;;UNGLQ
ORMQR;;;ORMQR;;;UNMQR;;;UNMQR
ORMLQ;;;ORMLQ;;;UNMLQ;;;UNMLQ
SGESV;;;DGESV;;;CGESV;;;ZGESV
SGETRF;;;DGETRF;;;CGETRF;;;ZGETRF
SGETRS;;;DGETRS;;;CGETRS;;;ZGETRS
SGEQRS;;;DGEQRS;;;CGEQRS;;;ZGEQRS
SGEQRF;;;DGEQRF;;;CGEQRF;;;ZGEQRF
SGELQF;;;DGELQF;;;CGELQF;;;ZGELQF
SGELQS;;;DGELQS;;;CGELQS;;;ZGELQS
STRSMPL;;;DTRSMPL;;;CTRSMPL;;;ZTRSMPL
STRSM;;;DTRSM;;;CTRSM;;;ZTRSM
SPOSV;;;DPOSV;;;CPOSV;;;ZPOSV
SPOTRF;;;DPOTRF;;;CPOTRF;;;ZPOTRF
SPOTRS;;;DPOTRS;;;CPOTRS;;;ZPOTRS
SGELS;;;DGELS;;;CGELS;;;ZGELS
Workspace_s;;;Workspace_d;;;Workspace_c;;;Workspace_z
workspace_s;;;workspace_d;;;workspace_c;;;workspace_z
syrk;;;syrk;;;herk;;;herk
**T;;;**T;;;**H;;;**H
"""

G_C_deletions = """
CBLAS_SADDR;;;CBLAS_SADDR;;; ;;;
"""

G_H_deletions = """
CBLAS_SADDR;;;CBLAS_SADDR;;; ;;;
"""

G_C_all_types_translations_before = """
PLASMA_desc;;;P@LASMA_@desc
PLASMA_Desc;;;P@LASMA_@Desc
PLASMA_context;;;P@LASMA_@context
PLASMA_Disable;;;P@LASMA_@Disable
"""

G_C_all_types_translations_after = """
P@LASMA_@desc;;;PLASMA_desc
P@LASMA_@Desc;;;PLASMA_Desc
P@LASMA_@context;;;PLASMA_context
P@LASMA_@Disable;;;PLASMA_Disable
"""

G_H_translations = """
float;;;double;;;PLASMA_Complex32_t;;;PLASMA_Complex64_t
float;;;double;;;float _Complex;;;double _Complex
float;;;double;;;float;;;double
PLASMA_sor;;;PLASMA_dor;;;PLASMA_cun;;;PLASMA_zun
PLASMA_S;;;PLASMA_D;;;PLASMA_C;;;PLASMA_Z
PLASMA_s;;;PLASMA_d;;;PLASMA_c;;;PLASMA_z
plasma_s;;;plasma_d;;;plasma_c;;;plasma_z
plasma_ps;;;plasma_pd;;;plasma_pc;;;plasma_pz
PlasmaRealFloat;;;PlasmaRealDouble;;;PlasmaComplexFloat;;;PlasmaComplexDouble
PlasmaTrans;;;PlasmaTrans;;;PlasmaConjTrans;;;PlasmaConjTrans
cblas_s;;;cblas_d;;;cblas_c;;;cblas_z
CORE_s;;;CORE_d;;;CORE_c;;;CORE_z
core_s;;;core_d;;;core_c;;;core_z
slamch;;;dlamch;;;slamch;;;dlamch
slange;;;dlange;;;clange;;;zlange
slansy;;;dlansy;;;clansy;;;zlansy
slacpy;;;dlacpy;;;clacpy;;;zlacpy
spotrf;;;dpotrf;;;cpotrf;;;zpotrf
ormqr;;;ormqr;;;unmqr;;;unmqr
ormlq;;;ormlq;;;unmlq;;;unmlq
orgqr;;;orgqr;;;ungqr;;;ungqr
orglq;;;orglq;;;unglq;;;unglq
SGESV;;;DGESV;;;CGESV;;;ZGESV
SGETRF;;;DGETRF;;;CGETRF;;;ZGETRF
SGETRS;;;DGETRS;;;CGETRS;;;ZGETRS
SGEQRF;;;DGEQRF;;;CGEQRF;;;ZGEQRF
SGELQF;;;DGELQF;;;CGELQF;;;ZGELQF
SORMQR;;;DORMQR;;;CUNMQR;;;ZUNMQR
SORMLQ;;;DORMLQ;;;CUNMLQ;;;ZUNMLQ
STRSMPL;;;DTRSMPL;;;CTRSMPL;;;ZTRSMPL
STRSM;;;DTRSM;;;CTRSM;;;ZTRSM
SPOSV;;;DPOSV;;;CPOSV;;;ZPOSV
SPOTRF;;;DPOTRF;;;CPOTRF;;;ZPOTRF
SPOTRS;;;DPOTRS;;;CPOTRS;;;ZPOTRS
SGELS;;;DGELS;;;CGELS;;;ZGELS
CORE_S;;;CORE_D;;;CORE_C;;;CORE_Z
Workspace_s;;;Workspace_d;;;Workspace_c;;;Workspace_z
syrk;;;syrk;;;herk;;;herk
"""


def get_replacement_list(idx, lang):
  global G_C_translations, G_F_translations, G_H_translations

  translations = {"C":G_C_translations, "F":G_F_translations, "H":G_H_translations}

  l = []
  for line in translations[lang].split("\n"):
    if len(line.strip()) < 1:
      continue
    l.append(line.strip().split(";;;")[idx].strip())

  return l

def get_deletion_list(idx, lang):
  global G_C_deletions, G_F_deletions

  deletions = {"C":G_C_deletions, "F":G_F_deletions, "H":G_H_deletions}

  l = []
  for line in deletions[lang].split("\n"):
    if len(line.strip()) < 1:
      continue
    l.append(line.strip().split(";;;")[idx].strip())

  return l

def get_global_variables(idx, lang):
  global G_F_Variables, G_C_Variables, G_H_Variables

  variables = {"C":G_C_Variables, "F":G_F_Variables, "H":G_H_Variables}

  d = dict()
  for line in variables[lang].split("\n"):
    if len(line.strip()) < 1:
      continue
    d[line.strip().split(";;;")[idx]] = 1

  return d

def get_replacemnts_before_and_after():
  global G_C_all_types_translations_before, G_C_all_types_translations_after

  lll = list()
  for var in (G_C_all_types_translations_before, G_C_all_types_translations_after):
    l = list()
    for line in var.split("\n"):
      if len(line.strip()) < 1:
        continue
      l.append(map(string.strip, line.strip().split(";;;")))
    lll.append(l)

  return lll[0], lll[1]

def unique_filename_part(lttr):
  return "_" + lttr

def get_current_precision_letter(fname, lang):
  all_lttrs = ("s", "d", "c", "z")

  idx = fname.find("core_")
  if idx < 0:
    idx = 0
  else:
    idx += 4 # skipe 'core'

  cur_lttr = ""
  for lttr in all_lttrs:
    s = unique_filename_part(lttr)
    if fname.find(s, idx) > 0:
      cur_lttr = lttr
      break

  if not cur_lttr:
    base_fname = os.path.basename(fname)
    for lttr in all_lttrs:
      if base_fname[0] == lttr:
        cur_lttr = lttr
        break
      elif "p" == base_fname[0] and base_fname[1] == lttr:
        cur_lttr = lttr
        break
      elif "x" == base_fname[0] and base_fname[1] == lttr:
        cur_lttr = lttr
        break

  return cur_lttr

def replace_or_un(filename):
  new_filename = filename
  # replace complex-orthognal with complex-unitary and real-unitary with real-orthogonal
  #for rp in (("psun", "psor"), ("pdun", "pdor"), ("pcor", "pcun"), ("pzor", "pzun")):
  for rp in (("sunmqr", "sormqr"), ("dunmqr", "dormqr"), ("cormqr", "cunmqr"), ("zormqr", "zunmqr"),
             ("sunmlq", "sormlq"), ("dunmlq", "dormlq"), ("cormlq", "cunmlq"), ("zormlq", "zunmlq"),
             ("sungqr", "sorgqr"), ("dungqr", "dorgqr"), ("corgqr", "cungqr"), ("zorgqr", "zungqr"),
			 ("sunglq", "sorglq"), ("dunglq", "dorglq"), ("corglq", "cunglq"), ("zorglq", "zunglq")):
    new_filename = new_filename.replace(rp[0], rp[1])

  return new_filename

def get_new_filename(fname, cur_lttr, lttr):
  cs = unique_filename_part(cur_lttr)
  if fname.find(cs) >= 0:
    s = unique_filename_part(lttr)
    return replace_or_un(fname.replace(cs, s))

  base_fname = os.path.basename(fname)
  new_filename = ""
  if base_fname[0] == cur_lttr:
    new_filename = fname[:-len(base_fname)] + lttr + base_fname[1:]
  elif base_fname[0] == "p" and base_fname[1] == cur_lttr:
    new_filename = fname[:-len(base_fname)+1] + lttr + base_fname[2:]
  elif base_fname[0] == "x" and base_fname[1] == cur_lttr:
    new_filename = fname[:-len(base_fname)+1] + lttr + base_fname[2:]

  return replace_or_un(new_filename)

def list_file_replace(list_file, replacement_list):
  new_lf = list()
  for line in list_file:
    new_line = line
    for orig, repl in replacement_list:
      new_line = new_line.replace(orig, repl)
    new_lf.append(new_line)
  return new_lf

def list_file_delete(list_file, deletion_list):
  new_lf = list()
  for line in list_file:
    new_line = line
    for del_str in deletion_list:
      new_line = new_line.replace(del_str, "")
    new_lf.append(new_line)
  return new_lf

def get_variable_value(var, global_variables):
  return global_variables.get(var, 0)

def check_conditional(line, global_variables):
  all_tokens = line.strip().split()

  if len(all_tokens) == 1:
    return get_variable_value(all_tokens[0], global_variables)

  elif len(all_tokens) == 3:
    expr1, op, expr2 = all_tokens
    if op == ".OR.":
      return get_variable_value(expr1, global_variables) or get_variable_value(expr2, global_variables)
    elif op == ".AND.":
      return get_variable_value(expr1, global_variables) and get_variable_value(expr2, global_variables)
    else:
      raise ValueError, "Cannot parse expression: %s" % line

  else:
    raise ValueError, "Cannot parse expression: %s" % line

def list_file_conditionals(list_file, global_variables):
  produce_output = 1
  suppress_line = 0
  ifdef_in = 0

  new_lf = list()
  for line in list_file:
    new_line = line

    if new_line.startswith("*#IFDEF") or new_line.startswith("//#IFDEF"):
      suppress_line = 1
      ifdef_in = 1
      if check_conditional(new_line[new_line.find("IFDEF")+5:], global_variables):
        produce_output = 1
      else:
        produce_output = 0

    elif new_line.startswith("*#ENDIF") or new_line.startswith("//#ENDIF"):
      if not ifdef_in:
        raise ValueError, "#ENDIF without #IFDEF"
      suppress_line = 1
      ifdef_in = 0
      produce_output = 1

    if produce_output and not suppress_line:
      new_lf.append(new_line)

    suppress_line = 0

  if ifdef_in:
    raise ValueError, "#IFDEF did not end with #ENDIF"

  return new_lf

def file_process(list_file, replacement_list, deletion_list, global_variables, replacements_before, replacements_after):
  new_list_file = list_file
  new_list_file = list_file_replace(new_list_file, replacements_before)
  new_list_file = list_file_conditionals(new_list_file, global_variables)
  new_list_file = list_file_replace(new_list_file, replacement_list)
  new_list_file = list_file_delete(new_list_file, deletion_list)
  new_list_file = list_file_replace(new_list_file, replacements_after)

  return new_list_file

def filename_replace(fname):
  lang = fname[-1].upper()
  if lang not in ("C", "F", "H"):
    raise ValueError

  all_lttrs = ("s", "d", "c", "z")

  replacements_before, replacements_after = get_replacemnts_before_and_after()
  replacement_dict = dict()
  deletion_dict = dict()
  variable_dict = dict()
  for idx, letter in enumerate(all_lttrs):
    replacement_dict[letter] = get_replacement_list(idx, lang)
    deletion_dict[letter] = get_deletion_list(idx, lang)
    variable_dict[letter] = get_global_variables(idx, lang)

  cur_lttr = get_current_precision_letter(fname, lang)

  if not cur_lttr:
    raise ValueError

  for lttr in all_lttrs:
    if lttr == cur_lttr:
      continue

    replacement_list = list()
    for idx, s in enumerate(replacement_dict[cur_lttr]):
      replacement_list.append((s, replacement_dict[lttr][idx]))

    new_fname = get_new_filename(fname, cur_lttr, lttr)
    if not new_fname:
      raise ValueError, "Failure with: %s %s %s" % (fname, cur_lttr, lttr)


    f = open(fname)
    list_file = []
    for line in f:
      list_file.append(line)
    f.close()

    new_list_file = file_process(list_file, replacement_list, deletion_dict[lttr], variable_dict[lttr], replacements_before, replacements_after)

    new_f = open(new_fname, "w")
    for line in new_list_file:
      new_f.write(line)
    new_f.close()

def main(argv):
  for file_name in argv[1:]:
    filename_replace(file_name)

  return 0

if "__main__" == __name__:
  sys.exit(main(sys.argv))
