/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zunmlq_Tile - overwrites the general M-by-N matrix C with Q*C, where Q is an orthogonal
// matrix (unitary in the complex case) defined as the product of elementary reflectors returned
// by PLASMA_zgelqf_Tile Q is of order M.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// side     PLASMA_enum (IN)
//          Intended usage:
//          = PlasmaLeft:  apply Q or Q**H from the left;
//          = PlasmaRight: apply Q or Q**H from the right.
//          Currently only PlasmaLeft is supported.
//
// trans    PLASMA_enum (IN)
//          Intended usage:
//          = PlasmaNoTrans:   no transpose, apply Q;
//          = PlasmaConjTrans: conjugate transpose, apply Q**H.
//          Currently only PlasmaConjTrans is supported.
//
// A        PLASMA_Complex64_t* (IN)
//          Details of the LQ factorization of the original matrix A as returned by PLASMA_zgelqf.
//
// T        PLASMA_Complex64_t* (IN)
//          Auxiliary factorization data, computed by PLASMA_zgelqf.
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the M-by-N matrix B.
//          On exit, B is overwritten by Q*B or Q**H*B.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zunmlq_Tile(PLASMA_enum side, PLASMA_enum trans, PLASMA_desc *A, PLASMA_desc *T, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descT = *T;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zunmlq_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zunmlq_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descT) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zunmlq_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zunmlq_Tile", "invalid third descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (side != PlasmaLeft) {
        plasma_error("PLASMA_zunmlq_Tile", "only PlasmaLeft supported");
        return PLASMA_ERR_NOT_SUPPORTED;
    }
    if (trans != PlasmaConjTrans) {
        plasma_error("PLASMA_zunmlq_Tile", "only PlasmaConjTrans supported");
        return PLASMA_ERR_NOT_SUPPORTED;
    }
    /* Quick return - currently NOT equivalent to LAPACK's:
     * CALL DLASET( 'Full', MAX( M, N ), NRHS, ZERO, ZERO, B, LDB ) */
/*
    if (min(M, min(N, K)) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_3(plasma_pzunmlq,
        PLASMA_desc, descA,
        PLASMA_desc, descB,
        PLASMA_desc, descT);

    return PLASMA_SUCCESS;
}
