/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_ztrsm_Tile - Computes triangular solve A*X = B or X*A = B
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// side     PLASMA_enum (IN)
//          Specifies whether A appears on the left or on the right of X:
//          = PlasmaLeft:  A*X = B
//          = PlasmaRight: X*A = B
//
// uplo     PLASMA_enum (IN)
//          Specifies whether the matrix A is upper triangular or lower triangular:
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// transA   PLASMA_enum (IN)
//          Specifies whether the matrix A is transposed, not transposed or conjugate transposed:
//          = PlasmaNoTrans:   A is transposed;
//          = PlasmaTrans:     A is not transposed;
//          = PlasmaConjTrans: A is conjugate transposed.
//
// diag     PLASMA_enum (IN)
//          Specifies whether or not A is unit triangular:
//          = PlasmaNonUnit: A is non unit;
//          = PlasmaUnit:    A us unit.
//
// A        PLASMA_Complex64_t* (IN)
//          The triangular matrix A. If uplo = PlasmaUpper, the leading N-by-N upper triangular
//          part of the array A contains the upper triangular matrix, and the strictly lower
//          triangular part of A is not referenced. If uplo = PlasmaLower, the leading N-by-N
//          lower triangular part of the array A contains the lower triangular matrix, and the
//          strictly upper triangular part of A is not referenced. If diag = PlasmaUnit, the
//          diagonal elements of A are also not referenced and are assumed to be 1.
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the N-by-NRHS right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_ztrsm_Tile(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA, PLASMA_enum diag, PLASMA_desc *A, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_ztrsm_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_ztrsm_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_ztrsm_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (side != PlasmaLeft && side != PlasmaRight) {
        plasma_error("PLASMA_ztrsm_Tile", "illegal value of side");
        return -1;
    }
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_ztrsm_Tile", "illegal value of uplo");
        return -2;
    }
    if (transA != PlasmaConjTrans && transA != PlasmaNoTrans) {
        plasma_error("PLASMA_ztrsm_Tile", "illegal value of transA");
        return -3;
    }
    if (diag != PlasmaUnit && diag != PlasmaNonUnit) {
        plasma_error("PLASMA_ztrsm_Tile", "illegal value of diag");
        return -4;
    }
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_7(plasma_pztrsm,
        PLASMA_enum, PlasmaLeft,
        PLASMA_enum, uplo,
        PLASMA_enum, transA,
        PLASMA_enum, diag,
        PLASMA_Complex64_t, 1.0,
        PLASMA_desc, descA,
        PLASMA_desc, descB);

    return PLASMA_SUCCESS;
}
