/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zpotrf_Tile - Computes the Cholesky factorization of a symmetric positive definite
// (or Hermitian positive definite in the complex case) matrix A.
// The factorization has the form
//
//   A = U**H * U, if uplo = PlasmaUpper, or
//   A = L * L**H, if uplo =  PlasmaLower,
//
// where U is an upper triangular matrix and L is a lower triangular matrix.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// uplo     PLASMA_enum (IN)
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the symmetric positive definite (or Hermitian) matrix A.
//          If uplo = PlasmaUpper, the leading N-by-N upper triangular part of A
//          contains the upper triangular part of the matrix A, and the strictly lower triangular
//          part of A is not referenced.
//          If UPLO = 'L', the leading N-by-N lower triangular part of A contains the lower
//          triangular part of the matrix A, and the strictly upper triangular part of A is not
//          referenced.
//          On exit, if return value = 0, the factor U or L from the Cholesky factorization
//          A = U**H*U or A = L*L**H.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          > 0: if i, the leading minor of order i of A is not positive definite, so the
//               factorization could not be completed, and the solution has not been computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zpotrf_Tile(PLASMA_enum uplo, PLASMA_desc *A)
{
    PLASMA_desc descA = *A;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zpotrf_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zpotrf_Tile", "invalid descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_zpotrf_Tile", "illegal value of uplo");
        return -1;
    }
    /* Quick return */
/*
    if (max(N, 0) == 0)
        return PLASMA_SUCCESS;
*/
    plasma_parallel_call_2(plasma_pzpotrf,
        PLASMA_enum, uplo,
        PLASMA_desc, descA);

    return PLASMA_INFO;
}
