/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zpotrf - Computes the Cholesky factorization of a symmetric positive definite
// (or Hermitian positive definite in the complex case) matrix A.
// The factorization has the form
//
//   A = U**H * U, if uplo = PlasmaUpper, or
//   A = L * L**H, if uplo =  PlasmaLower,
//
// where U is an upper triangular matrix and L is a lower triangular matrix.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// uplo     PLASMA_enum (IN)
//          = PlasmaUpper: Upper triangle of A is stored;
//          = PlasmaLower: Lower triangle of A is stored.
//
// N        int (IN)
//          The order of the matrix A. N >= 0.
//
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the symmetric positive definite (or Hermitian) matrix A.
//          If uplo = PlasmaUpper, the leading N-by-N upper triangular part of A
//          contains the upper triangular part of the matrix A, and the strictly lower triangular
//          part of A is not referenced.
//          If UPLO = 'L', the leading N-by-N lower triangular part of A contains the lower
//          triangular part of the matrix A, and the strictly upper triangular part of A is not
//          referenced.
//          On exit, if return value = 0, the factor U or L from the Cholesky factorization
//          A = U**H*U or A = L*L**H.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value
//          > 0: if i, the leading minor of order i of A is not positive definite, so the
//               factorization could not be completed, and the solution has not been computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zpotrf(PLASMA_enum uplo, int N, PLASMA_Complex64_t *A, int LDA)
{
    int NB, NT;
    int status;
    PLASMA_Complex64_t *Abdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zpotrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (uplo != PlasmaUpper && uplo != PlasmaLower) {
        plasma_error("PLASMA_zpotrf", "illegal value of uplo");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_zpotrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, N)) {
        plasma_error("PLASMA_zpotrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (max(N, 0) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_ZPOSV, N, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zpotrf", "plasma_tune() failed");
        return status;
    }

    /* Set NT */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, NT*NT*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    if (Abdl == NULL) {
        plasma_error("PLASMA_zpotrf", "plasma_shared_alloc() failed");
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        N, N, 0, 0, N, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex64_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_2(plasma_pzpotrf,
        PLASMA_enum, uplo,
        PLASMA_desc, descA);

    if (PLASMA_INFO == PLASMA_SUCCESS)
    {
        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex64_t*, A,
            int, LDA);
    }
    plasma_shared_free(plasma, Abdl);
    return PLASMA_INFO;
}
