/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgetrf - Computes an LU factorization of a general M-by-N matrix A
// using the tile LU algorithm with partial tile pivoting with row interchanges.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A. N >= 0.
//
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the M-by-N matrix to be factored.
//          On exit, the tile factors L and U from the factorization.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// L        PLASMA_Complex64_t* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          required by PLASMA_zgetrs to solve the system of equations.
//
// IPIV     int* (OUT)
//          The pivot indices that define the permutations (not equivalent to LAPACK).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, and division by zero will occur
//               if it is used to solve a system of equations.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zgetrf(int M, int N, PLASMA_Complex64_t *A, int LDA, PLASMA_Complex64_t *L, int *IPIV)
{
    int NB, MT, NT;
    int status;
    PLASMA_Complex64_t *Abdl;
    PLASMA_Complex64_t *Lbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgetrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (M < 0) {
        plasma_error("PLASMA_zgetrf", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_zgetrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_zgetrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_ZGESV, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgetrf", "plasma_tune() failed");
        return status;
    }

    /* Set NT & NTRHS */
    NB = PLASMA_NB;
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    Lbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);
    if (Abdl == NULL || Lbdl == NULL) {
        plasma_error("PLASMA_zgetrf", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Lbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        M, N, 0, 0, M, N);

    PLASMA_desc descL = plasma_desc_init(
        Lbdl, PlasmaComplexDouble,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        M, N, 0, 0, M, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex64_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    /* Clear IPIV and Lbdl */
    plasma_memzero(IPIV, MT*NT*PLASMA_NB, PlasmaInteger);
    plasma_memzero(Lbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);
    /* Set INFO to SUCCESS */
    PLASMA_INFO = PLASMA_SUCCESS;

    plasma_parallel_call_3(plasma_pzgetrf,
        PLASMA_desc, descA,
        PLASMA_desc, descL,
        int*, IPIV);

    if (PLASMA_INFO == PLASMA_SUCCESS)
    {
        /* Return L to the user */
        plasma_memcpy(L, Lbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex64_t*, A,
            int, LDA);
    }

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Lbdl);
    return PLASMA_INFO;
}
