/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgesv_Tile - Computes the solution to a system of linear equations A * X = B,
// where A is an N-by-N matrix and X and B are N-by-NRHS matrices.
// The tile LU decomposition with partial tile pivoting and row interchanges is used to factor A.
// The factored form of A is then used to solve the system of equations A * X = B.
// All matrices are passed through descriptors. All dimensions are taken from the descriptors.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the N-by-N coefficient matrix A.
//          On exit, the tile L and U factors from the factorization (not equivalent to LAPACK).
//
// L        PLASMA_Complex64_t* (OUT)
//          On exit, auxiliary factorization data, related to the tile L factor,
//          necessary to solve the system of equations.
//
// IPIV     int* (OUT)
//          On exit, the pivot indices that define the permutations (not equivalent to LAPACK).
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the N-by-NRHS matrix of right hand side matrix B.
//          On exit, if return value = 0, the N-by-NRHS solution matrix X.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          > 0: if i, U(i,i) is exactly zero. The factorization has been completed,
//               but the factor U is exactly singular, so the solution could not be computed.

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zgesv_Tile(PLASMA_desc *A, PLASMA_desc *L, int *IPIV, PLASMA_desc *B)
{
    PLASMA_desc descA = *A;
    PLASMA_desc descL = *L;
    PLASMA_desc descB = *B;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgesv_Tile", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check descriptors for correctness */
    if (plasma_desc_check(&descA) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgesv_Tile", "invalid first descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descL) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgesv_Tile", "invalid second descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (plasma_desc_check(&descB) != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgesv_Tile", "invalid third descriptor");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Check input arguments */
    /* Quick return */
/*
    if (min(N, NRHS) == 0)
        return PLASMA_SUCCESS;
*/
    /* Clear IPIV and Lbdl */
    plasma_memzero(IPIV, descA.mt*descA.nt*PLASMA_NB, PlasmaInteger);
    plasma_memzero(descL.mat, descL.mt*descL.nt*PLASMA_IBNBSIZE, PlasmaComplexDouble);

    /* Set INFO to ZERO */
    PLASMA_INFO = PLASMA_SUCCESS;

    plasma_parallel_call_3(plasma_pzgetrf,
        PLASMA_desc, descA,
        PLASMA_desc, descL,
        int*, IPIV);

    if (PLASMA_INFO == PLASMA_SUCCESS)
    {
        plasma_parallel_call_4(plasma_pztrsmpl,
            PLASMA_desc, descA,
            PLASMA_desc, descB,
            PLASMA_desc, descL,
            int*, IPIV);

        plasma_parallel_call_7(plasma_pztrsm,
            PLASMA_enum, PlasmaLeft,
            PLASMA_enum, PlasmaUpper,
            PLASMA_enum, PlasmaNoTrans,
            PLASMA_enum, PlasmaNonUnit,
            PLASMA_Complex64_t, 1.0,
            PLASMA_desc, descA,
            PLASMA_desc, descB);
    }
    return PLASMA_INFO;
}
