/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgeqrf - Computes the tile QR factorization of a complex M-by-N matrix A: A = Q * R.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A.  N >= 0.
//
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit, the elements on and above the diagonal of the array contain the min(M,N)-by-N
//          upper trapezoidal matrix R (R is upper triangular if M >= N); the elements below the
//          diagonal represent the unitary matrix Q as a product of elementary reflectors stored
//          by tiles.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// T        PLASMA_Complex64_t* (OUT)
//          On exit, auxiliary factorization data, required by PLASMA_zgeqrs to solve the system
//          of equations.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zgeqrf(int M, int N, PLASMA_Complex64_t *A, int LDA, PLASMA_Complex64_t *T)
{
    int NB, MT, NT;
    int status;
    PLASMA_Complex64_t *Abdl;
    PLASMA_Complex64_t *Tbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgeqrf", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (M < 0) {
        plasma_error("PLASMA_zgeqrf", "illegal value of M");
        return -1;
    }
    if (N < 0) {
        plasma_error("PLASMA_zgeqrf", "illegal value of N");
        return -2;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_zgeqrf", "illegal value of LDA");
        return -4;
    }
    /* Quick return */
    if (min(M, N) == 0)
        return PLASMA_SUCCESS;

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_ZGELS, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgeqrf", "plasma_tune() failed");
        return status;
    }

    /* Set MT & NT */
    NB = PLASMA_NB;
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    Tbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);
    if (Abdl == NULL || Tbdl == NULL) {
        plasma_error("PLASMA_zgeqrf", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Tbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    PLASMA_desc descA = plasma_desc_init(
        Abdl, PlasmaComplexDouble,
        PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
        M, N, 0, 0, M, N);

    PLASMA_desc descT = plasma_desc_init(
        Tbdl, PlasmaComplexDouble,
        PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
        M, N, 0, 0, M, N);

    plasma_parallel_call_3(plasma_lapack_to_tile,
        PLASMA_Complex64_t*, A,
        int, LDA,
        PLASMA_desc, descA);

    plasma_parallel_call_2(plasma_pzgeqrf,
        PLASMA_desc, descA,
        PLASMA_desc, descT);

    /* Return T to the user */
    plasma_memcpy(T, Tbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);

    plasma_parallel_call_3(plasma_tile_to_lapack,
        PLASMA_desc, descA,
        PLASMA_Complex64_t*, A,
        int, LDA);

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Tbdl);
    return PLASMA_SUCCESS;
}
