/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA computational routine (version 2.0.0)                       ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_zgels - solves overdetermined or underdetermined linear systems involving an M-by-N
// matrix A using the QR or the LQ factorization of A.  It is assumed that A has full rank.
// The following options are provided:
//
// 1. trans = PlasmaNoTrans and M >= N: find the least squares solution of an overdetermined
//    system, i.e., solve the least squares problem: minimize || B - A*X ||.
//
// 2. trans = PlasmaNoTrans and M < N:  find the minimum norm solution of an underdetermined
//    system A * X = B.
//
// Several right hand side vectors B and solution vectors X can be handled in a single call;
// they are stored as the columns of the M-by-NRHS right hand side matrix B and the N-by-NRHS
// solution
// matrix X.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// trans    PLASMA_enum (IN)
//          Intended usage:
//          = PlasmaNoTrans:   the linear system involves A;
//          = PlasmaConjTrans: the linear system involves A**H.
//          Currently only PlasmaNoTrans is supported.
//
// M        int (IN)
//          The number of rows of the matrix A. M >= 0.
//
// N        int (IN)
//          The number of columns of the matrix A. N >= 0.
//
// NRHS     int (IN)
//          The number of right hand sides, i.e., the number of columns of the matrices B and X.
//          NRHS >= 0.
//
// A        PLASMA_Complex64_t* (INOUT)
//          On entry, the M-by-N matrix A.
//          On exit,
//          if M >= N, A is overwritten by details of its QR factorization as returned by
//                     PLASMA_zgeqrf;
//          if M < N, A is overwritten by details of its LQ factorization as returned by
//                      PLASMA_zgelqf.
//
// LDA      int (IN)
//          The leading dimension of the array A. LDA >= max(1,M).
//
// T        PLASMA_Complex64_t* (OUT)
//          On exit, auxiliary factorization data.
//
// B        PLASMA_Complex64_t* (INOUT)
//          On entry, the M-by-NRHS matrix B of right hand side vectors, stored columnwise;
//          On exit, if return value = 0, B is overwritten by the solution vectors, stored
//          columnwise:
//          if M >= N, rows 1 to N of B contain the least squares solution vectors; the residual
//          sum of squares for the solution in each column is given by the sum of squares of the
//          modulus of elements N+1 to M in that column;
//          if M < N, rows 1 to N of B contain the minimum norm solution vectors;
//
// LDB      int (IN)
//          The leading dimension of the array B. LDB >= MAX(1,M,N).

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = 0: successful exit
//          < 0: if -i, the i-th argument had an illegal value

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
#include "common.h"

int PLASMA_zgels(PLASMA_enum trans, int M, int N, int NRHS, PLASMA_Complex64_t *A, int LDA,
                 PLASMA_Complex64_t *T, PLASMA_Complex64_t *B, int LDB)
{
    int i, j;
    int NB, MT, NT, NTRHS;
    int status;
    PLASMA_Complex64_t *Abdl;
    PLASMA_Complex64_t *Bbdl;
    PLASMA_Complex64_t *Tbdl;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_zgels", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Check input arguments */
    if (trans != PlasmaNoTrans) {
        plasma_error("PLASMA_zgels", "only PlasmaNoTrans supported");
        return PLASMA_ERR_NOT_SUPPORTED;
    }
    if (M < 0) {
        plasma_error("PLASMA_zgels", "illegal value of M");
        return -2;
    }
    if (N < 0) {
        plasma_error("PLASMA_zgels", "illegal value of N");
        return -3;
    }
    if (NRHS < 0) {
        plasma_error("PLASMA_zgels", "illegal value of NRHS");
        return -4;
    }
    if (LDA < max(1, M)) {
        plasma_error("PLASMA_zgels", "illegal value of LDA");
        return -6;
    }
    if (LDB < max(1, max(M, N))) {
        plasma_error("PLASMA_zgels", "illegal value of LDB");
        return -9;
    }
    /* Quick return */
    if (min(M, min(N, NRHS)) == 0) {
        for (i = 0; i < max(M, N); i++)
            for (j = 0; j < NRHS; j++)
                B[j*LDB+i] = 0.0;
        return PLASMA_SUCCESS;
    }

    /* Tune NB & IB depending on M, N & NRHS; Set NBNBSIZE */
    status = plasma_tune(PLASMA_FUNC_ZGELS, M, N, NRHS);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_zgels", "plasma_tune() failed");
        return status;
    }

    /* Set MT, NT & NTRHS */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    NTRHS = (NRHS%NB==0) ? (NRHS/NB) : (NRHS/NB+1);

    /* Allocate memory for matrices in block layout */
    Abdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    Tbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);
    Bbdl = (PLASMA_Complex64_t *)plasma_shared_alloc(plasma, max(MT, NT)*NTRHS*PLASMA_NBNBSIZE, PlasmaComplexDouble);
    if (Abdl == NULL || Tbdl == NULL || Bbdl == NULL) {
        plasma_error("PLASMA_zgels", "plasma_shared_alloc() failed");
        plasma_shared_free(plasma, Abdl);
        plasma_shared_free(plasma, Tbdl);
        plasma_shared_free(plasma, Bbdl);
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }

    if (M >= N)
    {
        PLASMA_desc descA = plasma_desc_init(
            Abdl, PlasmaComplexDouble,
            PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
            M, N, 0, 0, M, N);

        PLASMA_desc descB = plasma_desc_init(
            Bbdl, PlasmaComplexDouble,
            PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
            M, NRHS, 0, 0, M, NRHS);

        PLASMA_desc descT = plasma_desc_init(
            Tbdl, PlasmaComplexDouble,
            PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
            M, N, 0, 0, M, N);

        plasma_parallel_call_3(plasma_lapack_to_tile,
            PLASMA_Complex64_t*, A,
            int, LDA,
            PLASMA_desc, descA);

        plasma_parallel_call_3(plasma_lapack_to_tile,
            PLASMA_Complex64_t*, B,
            int, LDB,
            PLASMA_desc, descB);

        plasma_parallel_call_2(plasma_pzgeqrf,
            PLASMA_desc, descA,
            PLASMA_desc, descT);

        /* Return T to the user */
        plasma_memcpy(T, Tbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);

        plasma_parallel_call_3(plasma_pzunmqr,
            PLASMA_desc, descA,
            PLASMA_desc, descB,
            PLASMA_desc, descT);

        plasma_parallel_call_7(plasma_pztrsm,
            PLASMA_enum, PlasmaLeft,
            PLASMA_enum, PlasmaUpper,
            PLASMA_enum, PlasmaNoTrans,
            PLASMA_enum, PlasmaNonUnit,
            PLASMA_Complex64_t, 1.0,
            PLASMA_desc, plasma_desc_submatrix(descA, 0, 0, N, N),
            PLASMA_desc, plasma_desc_submatrix(descB, 0, 0, N, NRHS));

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex64_t*, A,
            int, LDA);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            PLASMA_Complex64_t*, B,
            int, LDB);
    }
    else
    {
        for (i = M; i < N; i++)
            for (j = 0; j < NRHS; j++)
                B[j*LDB+i] = 0.0;

        PLASMA_desc descA = plasma_desc_init(
            Abdl, PlasmaComplexDouble,
            PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
            M, N, 0, 0, M, N);

        PLASMA_desc descB = plasma_desc_init(
            Bbdl, PlasmaComplexDouble,
            PLASMA_NB, PLASMA_NB, PLASMA_NBNBSIZE,
            N, NRHS, 0, 0, N, NRHS);

        PLASMA_desc descT = plasma_desc_init(
            Tbdl, PlasmaComplexDouble,
            PLASMA_IB, PLASMA_NB, PLASMA_IBNBSIZE,
            M, N, 0, 0, M, N);

        plasma_parallel_call_3(plasma_lapack_to_tile,
            PLASMA_Complex64_t*, A,
            int, LDA,
            PLASMA_desc, descA);

        plasma_parallel_call_3(plasma_lapack_to_tile,
            PLASMA_Complex64_t*, B,
            int, LDB,
            PLASMA_desc, descB);

        plasma_parallel_call_2(plasma_pzgelqf,
            PLASMA_desc, descA,
            PLASMA_desc, descT);

        /* Return T to the user */
        plasma_memcpy(T, Tbdl, MT*NT*PLASMA_IBNBSIZE, PlasmaComplexDouble);

        plasma_parallel_call_7(plasma_pztrsm,
            PLASMA_enum, PlasmaLeft,
            PLASMA_enum, PlasmaLower,
            PLASMA_enum, PlasmaNoTrans,
            PLASMA_enum, PlasmaNonUnit,
            PLASMA_Complex64_t, 1.0,
            PLASMA_desc, plasma_desc_submatrix(descA, 0, 0, M, M),
            PLASMA_desc, plasma_desc_submatrix(descB, 0, 0, M, NRHS));

        plasma_parallel_call_3(plasma_pzunmlq,
            PLASMA_desc, descA,
            PLASMA_desc, descB,
            PLASMA_desc, descT);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descA,
            PLASMA_Complex64_t*, A,
            int, LDA);

        plasma_parallel_call_3(plasma_tile_to_lapack,
            PLASMA_desc, descB,
            PLASMA_Complex64_t*, B,
            int, LDB);
    }

    plasma_shared_free(plasma, Abdl);
    plasma_shared_free(plasma, Tbdl);
    plasma_shared_free(plasma, Bbdl);
    return PLASMA_SUCCESS;
}
