/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.0.0)                          ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"
#include "allocate.h"
#include "auxiliary.h"
#include "workspace.h"

#include <stdlib.h>
#include <malloc.h>

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
int plasma_alloc_ibnb(int M, int N, int operation, int type, void **memptr)
{
    size_t size;
    int status;
    int NB, MT, NT;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("plasma_alloc_ibnb", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Tune NB & IB depending on M & N; Set IBNBSIZE */
    status = plasma_tune(operation, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("plasma_alloc_ibnb", "plasma_tune() failed");
        return PLASMA_ERR_UNEXPECTED;
    }
    /* Set MT & NT & allocate */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    size = MT*NT * PLASMA_IBNBSIZE * plasma_element_size(type);
    if (size <= 0) {
        *memptr = NULL;
        return PLASMA_SUCCESS;
    }
    //status = posix_memalign(memptr, STANDARD_PAGE_SIZE, size);
    *memptr = malloc(size);
  //if (status != 0) {
    if (*memptr == NULL) {
        plasma_error("plasma_alloc_ibnb", "posix_memalign() failed");
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }
    return PLASMA_SUCCESS;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
int plasma_alloc_ipiv(int M, int N, int operation, void **memptr)
{
    size_t size;
    int status;
    int NB, MT, NT;
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("plasma_alloc_ipiv", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Tune NB & IB depending on M & N; Set IBNBSIZE */
    status = plasma_tune(operation, M, N, 0);
    if (status != PLASMA_SUCCESS) {
        plasma_error("plasma_alloc_ipiv", "plasma_tune() failed");
        return PLASMA_ERR_UNEXPECTED;
    }
    /* Set MT & NT & allocate */
    NB = PLASMA_NB;
    NT = (N%NB==0) ? (N/NB) : (N/NB+1);
    MT = (M%NB==0) ? (M/NB) : (M/NB+1);
    size = MT*NT * NB * sizeof(int);
    if (size <= 0) {
        *memptr = NULL;
        return PLASMA_SUCCESS;
    }
  //status = posix_memalign(memptr, CACHE_LINE_SIZE, size);
    *memptr = malloc(size);
  //if (status != 0) {
    if (*memptr == NULL) {
        plasma_error("plasma_alloc_ipiv", "posix_memalign() failed");
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }
    return PLASMA_SUCCESS;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
int PLASMA_Dealloc_Handle(void **handle)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("PLASMA_Dealloc_Handle", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    if (*handle == NULL) {
        plasma_error("PLASMA_Dealloc_Handle", "attempting to deallocate a NULL handle");
        return PLASMA_ERR_UNALLOCATED;
    }
    free(*handle);
    *handle = NULL;
    return PLASMA_SUCCESS;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Complex Double
int PLASMA_Alloc_Workspace_zgels(int M, int N, PLASMA_Complex64_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_ZGELS, PlasmaComplexDouble, (void**)T); }

int PLASMA_Alloc_Workspace_zgeqrf(int M, int N, PLASMA_Complex64_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_ZGELS, PlasmaComplexDouble, (void**)T); }

int PLASMA_Alloc_Workspace_zgelqf(int M, int N, PLASMA_Complex64_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_ZGELS, PlasmaComplexDouble, (void**)T); }

int PLASMA_Alloc_Workspace_zgesv(int N, PLASMA_Complex64_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(N, N, PLASMA_FUNC_ZGESV, PlasmaComplexDouble, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_ZGESV, (void**)IPIV); }

int PLASMA_Alloc_Workspace_zgetrf(int M, int N, PLASMA_Complex64_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(M, N, PLASMA_FUNC_ZGESV, PlasmaComplexDouble, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(M, N, PLASMA_FUNC_ZGESV, (void**)IPIV); }

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Real Double
int PLASMA_Alloc_Workspace_dgels(int M, int N, double **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_DGELS, PlasmaRealDouble, (void**)T); }

int PLASMA_Alloc_Workspace_dgeqrf(int M, int N, double **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_DGELS, PlasmaRealDouble, (void**)T); }

int PLASMA_Alloc_Workspace_dgelqf(int M, int N, double **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_DGELS, PlasmaRealDouble, (void**)T); }

int PLASMA_Alloc_Workspace_dgesv(int N, double **L, int **IPIV) {
    int status = plasma_alloc_ibnb(N, N, PLASMA_FUNC_DGESV, PlasmaRealDouble, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_DGESV, (void**)IPIV); }

int PLASMA_Alloc_Workspace_dgetrf(int M, int N, double **L, int **IPIV) {
    int status = plasma_alloc_ibnb(M, N, PLASMA_FUNC_DGESV, PlasmaRealDouble, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(M, N, PLASMA_FUNC_DGESV, (void**)IPIV); }

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Complex Single
int PLASMA_Alloc_Workspace_cgels(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

int PLASMA_Alloc_Workspace_cgeqrf(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

int PLASMA_Alloc_Workspace_cgelqf(int M, int N, PLASMA_Complex32_t **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGELS, PlasmaComplexFloat, (void**)T); }

int PLASMA_Alloc_Workspace_cgesv(int N, PLASMA_Complex32_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(N, N, PLASMA_FUNC_CGESV, PlasmaComplexFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_CGESV, (void**)IPIV); }

int PLASMA_Alloc_Workspace_cgetrf(int M, int N, PLASMA_Complex32_t **L, int **IPIV) {
    int status = plasma_alloc_ibnb(M, N, PLASMA_FUNC_CGESV, PlasmaComplexFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(M, N, PLASMA_FUNC_CGESV, (void**)IPIV); }

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Real Single
int PLASMA_Alloc_Workspace_sgels(int M, int N, float **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_SGELS, PlasmaRealFloat, (void**)T); }

int PLASMA_Alloc_Workspace_sgeqrf(int M, int N, float **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_SGELS, PlasmaRealFloat, (void**)T); }

int PLASMA_Alloc_Workspace_sgelqf(int M, int N, float **T) {
    return plasma_alloc_ibnb(M, N, PLASMA_FUNC_SGELS, PlasmaRealFloat, (void**)T); }

int PLASMA_Alloc_Workspace_sgesv(int N, float **L, int **IPIV) {
    int status = plasma_alloc_ibnb(N, N, PLASMA_FUNC_SGESV, PlasmaRealFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(N, N, PLASMA_FUNC_SGESV, (void**)IPIV); }

int PLASMA_Alloc_Workspace_sgetrf(int M, int N, float **L, int **IPIV) {
    int status = plasma_alloc_ibnb(M, N, PLASMA_FUNC_SGESV, PlasmaRealFloat, (void**)L);
    if (status != PLASMA_SUCCESS)
        return status;
    return plasma_alloc_ipiv(M, N, PLASMA_FUNC_SGESV, (void**)IPIV); }
