/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.0.0)                          ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"
#include "auxiliary.h"
#include "descriptor.h"

#include <stdlib.h>

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Internal static descriptor initializer
PLASMA_desc plasma_desc_init(void *mat, PLASMA_enum dtyp, int mb, int nb, int bsiz, int lm, int ln,
                             int i, int j, int m, int n)
{
    PLASMA_desc desc;

    // Matrix address
    desc.mat = mat;
    // Matrix properties
    desc.dtyp = dtyp;
    desc.mb = mb;
    desc.nb = nb;
    desc.bsiz = bsiz;
    // Large matrix parameters
    desc.lm = lm;
    desc.ln = ln;
    // Large matrix derived parameters
    desc.lmt = (lm%nb==0) ? (lm/nb) : (lm/nb+1);
    desc.lnt = (ln%nb==0) ? (ln/nb) : (ln/nb+1);
    // Submatrix parameters
    desc.i = i;
    desc.j = j;
    desc.m = m;
    desc.n = n;
    // Submatrix derived parameters
    desc.mt = (i+m-1)/nb - i/nb + 1;
    desc.nt = (j+n-1)/nb - j/nb + 1;
    return desc;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Internal static descriptor initializer for submatrices
PLASMA_desc plasma_desc_submatrix(PLASMA_desc descA, int i, int j, int m, int n)
{
    PLASMA_desc descB;
    int nb;

    descB = descA;
    nb = descA.nb;
    // Submatrix parameters
    descB.i = i;
    descB.j = i;
    descB.m = m;
    descB.n = n;
    // Submatrix derived parameters
    descB.mt = (m%nb==0 && i%nb==0) ? (m/nb) : (m/nb+1);
    descB.nt = (n%nb==0 && j%nb==0) ? (n/nb) : (n/nb+1);
    return descB;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Check for descriptor correctness
int plasma_desc_check(PLASMA_desc *desc)
{
    if (desc->mat == NULL) {
        plasma_error("plasma_desc_check", "NULL matrix pointer");
        return PLASMA_ERR_UNALLOCATED;
    }
    if (desc->dtyp != PlasmaRealFloat &&
        desc->dtyp != PlasmaRealDouble &&
        desc->dtyp != PlasmaComplexFloat &&
        desc->dtyp != PlasmaComplexDouble  ) {
        plasma_error("plasma_desc_check", "invalid matrix type");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (desc->mb <= 0 || desc->nb <= 0) {
        plasma_error("plasma_desc_check", "negative tile dimension");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (desc->bsiz < desc->mb*desc->nb) {
        plasma_error("plasma_desc_check", "tile memory size smaller than the product of dimensions");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (desc->lm <= 0 || desc->ln <= 0) {
        plasma_error("plasma_desc_check", "negative matrix dimension");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (desc->i >= desc->lm || desc->j >= desc->ln) {
        plasma_error("plasma_desc_check", "beginning of the matrix out of scope");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    if (desc->i+desc->m > desc->lm || desc->j+desc->n > desc->ln) {
        plasma_error("plasma_desc_check", "submatrix out of scope");
        return PLASMA_ERR_ILLEGAL_VALUE;
    }
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Desc_Create - Create matrix descriptor.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// desc     PLASMA_desc** (OUT)
//          On exit, descriptor of the matrix.
//
// mat      void* (IN)
//          Memory location of the matrix.
//
// dtyp     PLASMA_enum (IN)
//          Data type of the matrix:
//          = PlasmaRealFloat:     single precision real (S),
//          = PlasmaRealDouble:    double precision real (D),
//          = PlasmaComplexFloat:  single precision complex (C),
//          = PlasmaComplexDouble: double precision complex (Z).
//
// mb       int (IN)
//          Number of rows in a tile.
//
// nb       int (IN)
//          Number of columns in a tile.
//
// bsiz     int (IN)
//          Nize in elements including padding.
//
// lm       int (IN)
//          Number of rows of the entire matrix.
//
// ln       int (IN)
//          Number of columns of the entire matrix.
//
// i        int (IN)
//          Row index to the beginning of the submatrix.
//
// j        int (IN)
//          Column indes to the beginning of the submatrix.
//
// m        int (IN)
//          Number of rows of the submatrix.
//
// n        int (IN)
//          Number of columns of the submatrix.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Desc_Create(PLASMA_desc **desc, void *mat, PLASMA_enum dtyp, int mb, int nb, int bsiz,
                       int lm, int ln, int i, int j, int m, int n)
{
    plasma_context_t *plasma;
    int status;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_error("PLASMA_Desc_Create", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    /* Allocate memory and initialize the descriptor */
    *desc = (PLASMA_desc*)malloc(sizeof(PLASMA_desc));
    if (*desc == NULL) {
        plasma_error("PLASMA_Desc_Create", "malloc() failed");
        return PLASMA_ERR_OUT_OF_RESOURCES;
    }
    **desc = plasma_desc_init(mat, dtyp, mb, nb, bsiz, lm, ln, i, j, m, n);
    status = plasma_desc_check(*desc);
    if (status != PLASMA_SUCCESS) {
        plasma_error("PLASMA_Desc_Create", "invalid descriptor");
        return status;
    }
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Desc_Destroy - Destroys matrix descriptor.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// desc     PLASMA_desc** (IN)
//          Matrix descriptor.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Desc_Destroy(PLASMA_desc **desc)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_error("PLASMA_Desc_Destroy", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    if (*desc == NULL) {
        plasma_error("PLASMA_Desc_Destroy", "attempting to destroy a NULL descriptor");
        return PLASMA_ERR_UNALLOCATED;
    }
    free(*desc);
    *desc = NULL;
    return PLASMA_SUCCESS;
}
