/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.0.0)                          ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#include "common.h"
#include "auxiliary.h"

#include <stdio.h>
#include <stdlib.h>

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  User's erroneous action without severe consequences.
//  Problems occuring while PLASMA is being used correctly.
//  Indicates a recoverable problem.
//  Context aware.
void plasma_warning(char *func_name, char* msg_text)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL)
        plasma_fatal_error("plasma_warning", "PLASMA not initialized");
    if (plasma->warnings_enabled)
        fprintf(stderr, "PLASMA WARNING: %s(): %s\n", func_name, msg_text);
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  User's erroneous action with potentially severe consequences.
//  Problems occuring due to incorrect use of PLASMA.
//  Indicates a recoverable problem.
//  Context aware.
void plasma_error(char *func_name, char* msg_text)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL)
        plasma_fatal_error("plasma_error", "PLASMA not initialized");
    if (plasma->errors_enabled)
        fprintf(stderr, "PLASMA ERROR: %s(): %s\n", func_name, msg_text);
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Unexpected behavior within the library.
//  Unrecoverable user errors.
//  Context oblivious.
void plasma_fatal_error(char *func_name, char* msg_text)
{
    fprintf(stderr, "PLASMA FATAL ERROR: %s(): %s\n", func_name, msg_text);
    exit(0);
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
int plasma_element_size(int type)
{
    switch(type) {
        case PlasmaByte:          return          1;
        case PlasmaInteger:       return   sizeof(int);
        case PlasmaRealFloat:     return   sizeof(float);
        case PlasmaRealDouble:    return   sizeof(double);
        case PlasmaComplexFloat:  return 2*sizeof(float);
        case PlasmaComplexDouble: return 2*sizeof(double);
        default: plasma_fatal_error("plasma_element_size", "undefined type");
                 return PLASMA_ERR_ILLEGAL_VALUE;

    }
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
void plasma_memcpy(void *dst, void *src, PLASMA_size size, int type)
{
    memcpy(dst, src, size*plasma_element_size(type));
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
void plasma_memzero(void *memptr, PLASMA_size size, int type)
{
    memset(memptr, 0, size*plasma_element_size(type));
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
void plasma_memset_int(int *mem, int size, int value)
{
    int i;

    for (i = 0; i < size; i++)
        mem[i] = value;
}

//* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Returns core id
int plasma_rank(plasma_context_t *plasma)
{
    int rank;
    pthread_t thread_id;

    thread_id = pthread_self();
    for (rank = 0; rank < plasma->world_size; rank++)
        if (plasma->thread_id[rank] == thread_id)
            return rank;
    return PLASMA_ERR_NOT_FOUND;
}

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Tune block size nb and internal block size ib
int plasma_tune(PLASMA_enum func, int M, int N, int NRHS)
{
    plasma_context_t *plasma;

    plasma = plasma_context_self();
    if (plasma == NULL) {
        plasma_fatal_error("plasma_tune", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }

    if (!plasma->autotuning_enabled)
        return PLASMA_SUCCESS;

    switch (func) {
        case PLASMA_FUNC_SPOSV:
        case PLASMA_FUNC_DPOSV:
        case PLASMA_FUNC_CPOSV:
        case PLASMA_FUNC_ZPOSV:
            plasma->nb = 120;
            plasma->ib = 120;  // not used in Cholesky
            break;
        case PLASMA_FUNC_SGELS:
        case PLASMA_FUNC_DGELS:
        case PLASMA_FUNC_CGELS:
        case PLASMA_FUNC_ZGELS:
            plasma->nb = 144;
            plasma->ib =  48;
            break;
        case PLASMA_FUNC_SGESV:
        case PLASMA_FUNC_DGESV:
        case PLASMA_FUNC_CGESV:
        case PLASMA_FUNC_ZGESV:
        case PLASMA_TUNE_ZCGESV:
        case PLASMA_TUNE_DSGESV:
            plasma->nb = 200;
            plasma->ib =  40;
            break;
        default:
            plasma_fatal_error("plasma_tune", "illegal parameter value");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
    /* Calculate A, B tile size and round up to cache line size */
    /* round up for the smallest type (float) - will hold for all */
    plasma->nbnbsize = plasma->nb * plasma->nb; // * sizeof(float);
//  plasma->nbnbsize = roundup(plasma->nbnbsize, CACHE_LINE_SIZE);
//  plasma->nbnbsize /= sizeof(float);
    /* Calculate T, L tile size and round up to cache line size */
    /* round up for the smallest type (float) - will hold for all */
    plasma->ibnbsize = plasma->ib * plasma->nb; // * sizeof(float);
//  plasma->ibnbsize = roundup(plasma->ibnbsize, CACHE_LINE_SIZE);
//  plasma->ibnbsize /= sizeof(float);
    return PLASMA_SUCCESS;
}

/* /////////////////////////// P /// U /// R /// P /// O /// S /// E /////////////////////////// */
// PLASMA_Version - Reports PLASMA version number.

/* ///////////////////// A /// R /// G /// U /// M /// E /// N /// T /// S ///////////////////// */
// ver_major    int (OUT)
//              PLASMA major version number.
//
// ver_minor    int (OUT)
//              PLASMA minor version number.
//
// ver_micro    int (OUT)
//              PLASMA micro version number.

/* ///////////// R /// E /// T /// U /// R /// N /////// V /// A /// L /// U /// E ///////////// */
//          = PLASMA_SUCCESS: successful exit

/* //////////////////////////////////// C /// O /// D /// E //////////////////////////////////// */
int PLASMA_Version(int *ver_major, int *ver_minor, int *ver_micro)
{
    if (! ver_major && ! ver_minor && ! ver_micro)
        return  PLASMA_ERR_ILLEGAL_VALUE;

    if (ver_major)
        *ver_major = PLASMA_VERSION_MAJOR;

    if (ver_minor)
        *ver_minor = PLASMA_VERSION_MINOR;

    if (ver_micro)
        *ver_micro = PLASMA_VERSION_MICRO;

    return PLASMA_SUCCESS;
}
