/* ///////////////////////////// P /// L /// A /// S /// M /// A /////////////////////////////// */
/* ///                    PLASMA auxiliary routines (version 2.0.0)                          ///
 * ///                    Release Date: July, 4th 2009                                       ///
 * ///                    PLASMA is a software package provided by Univ. of Tennessee,       ///
 * ///                    Univ. of California Berkeley and Univ. of Colorado Denver          /// */
/* ///////////////////////////////////////////////////////////////////////////////////////////// */
#ifndef _PLASMA_H_
#define _PLASMA_H_

#define PLASMA_VERSION_MAJOR 2
#define PLASMA_VERSION_MINOR 0
#define PLASMA_VERSION_MICRO 0

/* CBLAS requires for scalar arguments to be passed by address rather than by value */
#define CBLAS_SADDR &

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA constants - precisions
#define PlasmaByte          0
#define PlasmaInteger       1
#define PlasmaRealFloat     2
#define PlasmaRealDouble    3
#define PlasmaComplexFloat  4
#define PlasmaComplexDouble 5

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA types
typedef int  PLASMA_enum;
typedef int  PLASMA_bool;
typedef long PLASMA_index;
typedef long PLASMA_size;
typedef float _Complex PLASMA_Complex32_t;
typedef double _Complex PLASMA_Complex64_t;

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Tile matrix descriptor
typedef struct plasma_desc_t {
    void *mat;          // pointer to the beginning of the matrix
    PLASMA_enum dtyp;   // precision of the matrix
    int mb;             // number of rows in a tile
    int nb;             // number of columns in a tile
    int bsiz;           // size in elements including padding
    int lm;             // number of rows of the entire matrix
    int ln;             // number of columns of the entire matrix
    int lmt;            // number of tile rows of the entire matrix - derived parameter
    int lnt;            // number of tile columns of the entire matrix - derived parameter
    int i;              // row index to the beginning of the submatrix
    int j;              // column indes to the beginning of the submatrix
    int m;              // number of rows of the submatrix
    int n;              // number of columns of the submatrix
    int mt;             // number of tile rows of the submatrix - derived parameter
    int nt;             // number of tile columns of the submatrix - derived parameter
} PLASMA_desc;

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA constants - CBLAS & LAPACK
#define PlasmaNoTrans       111
#define PlasmaTrans         112
#define PlasmaConjTrans     113

#define PlasmaUpper         121
#define PlasmaLower         122

#define PlasmaNonUnit       131
#define PlasmaUnit          132

#define PlasmaLeft          141
#define PlasmaRight         142

#define PlasmaForward       391
#define PlasmaBackward      392

#define PlasmaColumnwise    401
#define PlasmaRowwise       402

#define plasma_const_neg(const) (((const-1)^0x01)+1)

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA constants - boolean
#define PLASMA_FALSE    0
#define PLASMA_TRUE     1

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  State machine switches
#define PLASMA_WARNINGS     1
#define PLASMA_ERRORS       2
#define PLASMA_AUTOTUNING   3

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA constants - configuration parameters
#define PLASMA_CONCURRENCY      1
#define PLASMA_TILE_SIZE        2
#define PLASMA_INNER_BLOCK_SIZE 3

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  PLASMA constants - success & error codes
#define PLASMA_SUCCESS                 0
#define PLASMA_ERR_NOT_INITIALIZED  -101
#define PLASMA_ERR_REINITIALIZED    -102
#define PLASMA_ERR_NOT_SUPPORTED    -103
#define PLASMA_ERR_ILLEGAL_VALUE    -104
#define PLASMA_ERR_NOT_FOUND        -105
#define PLASMA_ERR_OUT_OF_RESOURCES -106
#define PLASMA_ERR_INTERNAL_LIMIT   -107
#define PLASMA_ERR_UNALLOCATED      -108
#define PLASMA_ERR_FILESYSTEM       -109
#define PLASMA_ERR_UNEXPECTED       -110

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  LAPACK Constants
extern char *plasma_lapack_constants[];
#define lapack_const(plasma_const) plasma_lapack_constants[plasma_const]

#ifdef __cplusplus
extern "C" {
#endif

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Math function prototypes
#include <plasma_z.h>
#include <plasma_d.h>
#include <plasma_c.h>
#include <plasma_s.h>
#include <plasma_m.h>

/* ///////////////////////////////////////////////////////////////////////////////////////////// */
//  Auxiliary function prototypes
int PLASMA_Version(int *ver_major, int *ver_minor, int *ver_micro);
int PLASMA_Enable(PLASMA_enum lever);
int PLASMA_Disable(PLASMA_enum lever);
int PLASMA_Set(PLASMA_enum param, int value);
int PLASMA_Get(PLASMA_enum param, int *value);
int PLASMA_Init(int cores);
int PLASMA_Finalize();
int PLASMA_Desc_Create(PLASMA_desc **desc, void *mat, PLASMA_enum dtyp, int mb, int nb, int bsiz, int lm, int ln, int i, int j, int m, int n);
int PLASMA_Desc_Destroy(PLASMA_desc **desc);
int PLASMA_Lapack_to_Tile(void *Af77, int LDA, PLASMA_desc *A);
int PLASMA_Tile_to_Lapack(PLASMA_desc *A, void *Af77, int LDA);

#ifdef __cplusplus
}
#endif

#endif
