/*////////////////////////////////////////////////////////////////////////////////////////
 *  -- PLASMA --
 *     University of Tennessee
 */
#ifndef _PLASMA_H_
#define _PLASMA_H_

#if defined(c_plusplus) || defined(__cplusplus)
extern "C" {
#endif

/*////////////////////////////////////////////////////////////////////////////////////////
 *  PLASMA types
 */
typedef int  PLASMA_enum;
typedef int  PLASMA_bool;
typedef long PLASMA_long;

/*////////////////////////////////////////////////////////////////////////////////////////
 *  PLASMA constants - CBLAS & LAPACK
 */
#define PlasmaNoTrans       111
#define PlasmaTrans         112
#define PlasmaConjTrans     113

#define PlasmaUpper         121
#define PlasmaLower         122

#define PlasmaNonUnit       131
#define PlasmaUnit          132

#define PlasmaLeft          141
#define PlasmaRight         142

#define PlasmaForward       391
#define PlasmaBackward      392

#define PlasmaColumnwise    401
#define PlasmaRowwise       402

#define plasma_const_neg(const) (((const-1)^0x01)+1)

/*////////////////////////////////////////////////////////////////////////////////////////
 *  PLASMA constants - boolean
 */
#define PLASMA_FALSE    0
#define PLASMA_TRUE     1

/*////////////////////////////////////////////////////////////////////////////////////////
 *  PLASMA constants - configuration parameters
 */
#define PLASMA_CONCURRENCY  1

/*////////////////////////////////////////////////////////////////////////////////////////
 *  PLASMA constants - success & error codes
 */
#define PLASMA_SUCCESS               0
#define PLASMA_ERR_NOT_INITIALIZED  -1
#define PLASMA_ERR_REINITIALIZED    -2
#define PLASMA_ERR_NOT_SUPPORTED    -3
#define PLASMA_ERR_ILLEGAL_VALUE    -4
#define PLASMA_ERR_NOT_FOUND        -5
#define PLASMA_ERR_OUT_OF_MEMORY    -6
#define PLASMA_ERR_INTERNAL_LIMIT   -7
#define PLASMA_ERR_UNALLOCATED      -8
#define PLASMA_ERR_FILESYSTEM       -9

/*////////////////////////////////////////////////////////////////////////////////////////
 *  LAPACK Constants
 */
extern char *plasma_lapack_constants[];

#define lapack_const(plasma_const) plasma_lapack_constants[plasma_const]

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Math function prototypes
 */
int plasma_Init(int M, int N, int NRHS);
int plasma_Finalize();

int plasma_DPOSV(PLASMA_enum uplo, int N, int NRHS, double *A, int LDA,
                 double *B, int LDB);

int plasma_DPOTRF(PLASMA_enum uplo, int N, double *A, int LDA);

int plasma_DPOTRS(PLASMA_enum uplo, int N, int NRHS, double *A, int LDA,
                  double *B, int LDB);

int plasma_DTRSM(PLASMA_enum side, PLASMA_enum uplo, PLASMA_enum transA,
                 PLASMA_enum diag, int N, int NRHS, double *A, int LDA,
                 double *B, int LDB);

int plasma_DGELS(PLASMA_enum trans, int M, int N, int NRHS, double *A,
                 int LDA, double *T, double *B, int LDB);

int plasma_DGEQRF(int M, int N, double *A, int LDA, double *T);

int plasma_DORMQR(PLASMA_enum side, PLASMA_enum trans, int M, int NRHS, int N,
                  double *A, int LDA, double *T, double *B, int LDB);

int plasma_DGESV(int N, int NRHS, double *A, int LDA, double *L, int *IPIV,
                 double *B, int LDB);

int plasma_DGETRF(int M, int N, double *A, int LDA, double *L, int *IPIV);

int plasma_DGETRS(int M, int NRHS, int N, double *A, int LDA, double *L, int *IPIV,
                  double *B, int LDB);

int plasma_DTRSMPL(int M, int NRHS, int N, double *A, int LDA, double *L, int *IPIV,
                   double *B, int LDB);

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Auxiliary function prototypes
 */
double *plasma_Allocate_T(int M, int N);
int plasma_Free_T(double *T);

double *plasma_Allocate_L(int M, int N);
int plasma_Free_L(double *L);

int *plasma_Allocate_IPIV(int M, int N);
int plasma_Free_IPIV(int *IPIV);

#if defined(c_plusplus) || defined(__cplusplus)
}
#endif

#endif
