/*////////////////////////////////////////////////////////////////////////////////////////
 *  -- PLASMA --
 *     University of Tennessee
 */
#ifndef _PLASMA_GLOBAL_H_
#define _PLASMA_GLOBAL_H_

#include "plasma.h"

#include <pthread.h>

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Configuration
 */
#define CORES_MAX           1024
#define CACHE_LINE_SIZE     128
#define SHELL_CMD_OUT_MAX   256
#define HUGE_PAGE_FILE_NAME "/huge/plasma_huge_page.bin"

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Action commands
 */
#define PLASMA_ACT_STAND_BY      0
#define PLASMA_ACT_FINALIZE      1

#define PLASMA_ACT_DGEQRF        2
#define PLASMA_ACT_DORMQR        3
#define PLASMA_ACT_DTRSM         4
#define PLASMA_ACT_DPOTRF        5
#define PLASMA_ACT_DGELQF        6
#define PLASMA_ACT_DGETRF        7
#define PLASMA_ACT_DTRSMPL       8

#define PLASMA_ACT_F77_TO_BDL    9
#define PLASMA_ACT_BDL_TO_F77   10

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Tuned functions
 */
#define PLASMA_TUNE_DGELS   1
#define PLASMA_TUNE_DPOSV   2
#define PLASMA_TUNE_DGESV   3

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Utility
 */
#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))
#define roundup(a, b) (b <= 0) ? (a) : (((a) + (b)-1) & ~((b)-1))

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Main control data structure
 */
struct plasma_cntrl_t {
    pthread_mutex_t action_mutex;   // master->workers action communication
    pthread_cond_t action_condt;
    PLASMA_enum action;

    PLASMA_bool initialized;        // initialization flag
    PLASMA_bool bdl_huge_pages;     // huge page support flag
    int NB_max;
    int NB_min;
    int IB_max;
    int NB;
    int IB;
    int NBNBSIZE;                   // tile size padded to cache line size
    int IBNBSIZE;                   // T tile size padded to cache line size
    int cores_max;
    int cores_num;
    int bdl_size_elems;
    int progress_size_elems;

    int core_num[CORES_MAX];
    pthread_t core_id[CORES_MAX];
    pthread_attr_t core_attr;
    pthread_t thread_id[CORES_MAX];

    int cache_line_size;
    int page_size;              // standard page size
    int huge_page_size;
    int huge_pages_free;
    int huge_pages_total;
    int huge_page_fmem;
};

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Auxiliary storage
 */
struct plasma_aux_t {
    double *bdl_mem;            // BDL storage
    volatile int *progress;     // progress table
    double *WORK[CORES_MAX];    // kernel workspace
    double *TAU[CORES_MAX];     // kernel workspace
};

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Master->workers communication of arguments
 */
struct plasma_args_t {
    PLASMA_enum trans;
    PLASMA_enum side;
    PLASMA_enum uplo;
    PLASMA_enum diag;
    int M;
    int N;
    int NRHS;
    int NB;
    int NBNBSIZE;
    int IBNBSIZE;
    int MT;
    int MTB;
    int NT;
    int NTRHS;
    int IB;
    int LDA;
    int LDB;
    double *F77;
    double *A;
    double *B;
    double *T;
    double *L;
    int *IPIV;
    int INFO;
};

#endif