/*////////////////////////////////////////////////////////////////////////////////////////
 *  -- PLASMA --
 *     University of Tennessee
 */
#include "common.h"
#include "core_control.h"
#include "bdl_convert.h"
#include "initialize.h"
#include "auxiliary.h"
#include "barrier.h"

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Main thread control
 */
void *plasma_parallel_section(void *core_id)
{
    int my_core_id = *((int*)core_id);
    int status;
    PLASMA_enum action;

    /* Parallel initializations */
    status = plasma_parallel_init(my_core_id);
    if (status != PLASMA_SUCCESS) {
        plasma_error("plasma_parallel_section", "plasma_parallel_init() failed");
        return (void*)status;
    }
    plasma_barrier(my_core_id, plasma_cntrl.cores_num);

    while(1) {
        pthread_mutex_lock(&plasma_cntrl.action_mutex);
        while ((action = plasma_cntrl.action) == PLASMA_ACT_STAND_BY)
            pthread_cond_wait(&plasma_cntrl.action_condt, &plasma_cntrl.action_mutex);
        pthread_mutex_unlock(&plasma_cntrl.action_mutex);
        plasma_barrier(my_core_id, plasma_cntrl.cores_num);

        switch (action) {
            case PLASMA_ACT_DGEQRF:
                plasma_pDGEQRF(plasma_args.M, plasma_args.N, plasma_args.A,
                               plasma_args.NB, plasma_args.NBNBSIZE, plasma_args.IBNBSIZE,
                               plasma_args.IB, plasma_args.MT, plasma_args.NT, plasma_args.T,
                               &plasma_args.INFO, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_DGELQF:
                plasma_pDGELQF(plasma_args.M, plasma_args.N, plasma_args.A,
                               plasma_args.NB, plasma_args.NBNBSIZE, plasma_args.IBNBSIZE,
                               plasma_args.IB, plasma_args.MT, plasma_args.NT, plasma_args.T,
                               &plasma_args.INFO, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_DORMQR:
                plasma_pDORMQR(plasma_args.M, plasma_args.NRHS, plasma_args.N,
                               plasma_args.A, plasma_args.NB, plasma_args.NBNBSIZE,
                               plasma_args.IBNBSIZE, plasma_args.IB, plasma_args.MT,
                               plasma_args.NTRHS, plasma_args.NT, plasma_args.T,
                               plasma_args.B, &plasma_args.INFO, plasma_cntrl.cores_num,
                               my_core_id);
                break;
            case PLASMA_ACT_DTRSM:
                plasma_pDTRSM(PlasmaLeft, plasma_args.uplo, plasma_args.trans, plasma_args.diag,
                              plasma_args.N, plasma_args.NRHS, 1.0, plasma_args.A, plasma_args.NB,
                              plasma_args.NBNBSIZE, plasma_args.NT, plasma_args.MT, plasma_args.B,
                              plasma_args.MTB, plasma_args.NTRHS, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_DPOTRF:
                plasma_pDPOTRF(plasma_args.uplo, plasma_args.N, plasma_args.A, plasma_args.NB,
                               plasma_args.NBNBSIZE, plasma_args.NT, &plasma_args.INFO,
                               plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_DGETRF:
                plasma_pDGETRF(plasma_args.M, plasma_args.N, plasma_args.A, plasma_args.NB,
                               plasma_args.NBNBSIZE, plasma_args.IBNBSIZE, plasma_args.IB,
                               plasma_args.MT, plasma_args.NT, plasma_args.L, plasma_args.IPIV,
                               &plasma_args.INFO, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_DTRSMPL:
                plasma_pDTRSMPL(plasma_args.M, plasma_args.NRHS, plasma_args.N, plasma_args.A,
                                 plasma_args.NB, plasma_args.NBNBSIZE, plasma_args.IBNBSIZE,
                                 plasma_args.IB, plasma_args.MT, plasma_args.NTRHS, plasma_args.NT,
                                 plasma_args.L, plasma_args.IPIV, plasma_args.B, &plasma_args.INFO,
                                 plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_F77_TO_BDL:
                plasma_lapack_to_bdl(plasma_args.F77, plasma_args.A, plasma_args.M, plasma_args.N,
                                     plasma_args.LDA, plasma_args.NB, plasma_args.MT, plasma_args.NT,
                                     plasma_args.NBNBSIZE, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_BDL_TO_F77:
                plasma_bdl_to_lapack(plasma_args.A, plasma_args.F77, plasma_args.M, plasma_args.N,
                                     plasma_args.LDA, plasma_args.NB, plasma_args.MT, plasma_args.NT,
                                     plasma_args.NBNBSIZE, plasma_cntrl.cores_num, my_core_id);
                break;
            case PLASMA_ACT_FINALIZE:
                return NULL;
            default:
                break;
        }
        plasma_barrier(my_core_id, plasma_cntrl.cores_num);
    }
    return NULL;
}
