/*////////////////////////////////////////////////////////////////////////////////////////
 *  -- PLASMA --
 *     University of Tennessee
 */
#include "common.h"
#include "auxiliary.h"

#include <stdio.h>

/*////////////////////////////////////////////////////////////////////////////////////////
 */
void plasma_warning(char *func_name, char* msg_text)
{
    fprintf(stderr, "PLASMA warning: %s(): %s\n", func_name, msg_text);
}

/*////////////////////////////////////////////////////////////////////////////////////////
 */
void plasma_error(char *func_name, char* msg_text)
{
    fprintf(stderr, "PLASMA error: %s(): %s\n", func_name, msg_text);
}

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Set PLASMA integer parameter
 */
int plasma_set_int(PLASMA_enum param, int value)
{
    if (!plasma_cntrl.initialized) {
        plasma_warning("plasma_set_int", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    switch (param) {
        case PLASMA_CONCURRENCY:
            if (value <= 0 || value > plasma_cntrl.cores_max) {
                plasma_warning("plasma_set_int", "illegal parameter value");
                return PLASMA_ERR_ILLEGAL_VALUE;
            }
            plasma_cntrl.cores_num = value;
            break;
        default:
            plasma_error("plasma_set_int", "illegal parameter value");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
    return PLASMA_SUCCESS;
}

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Get PLASMA integer parameter
 */
int plasma_get_int(PLASMA_enum param)
{
    if (!plasma_cntrl.initialized) {
        plasma_warning("plasma_get_int", "PLASMA not initialized");
        return PLASMA_ERR_NOT_INITIALIZED;
    }
    switch (param) {
        case PLASMA_CONCURRENCY:
            return plasma_cntrl.cores_num;
        default:
            plasma_error("plasma_get_int", "illegal parameter value");
            return PLASMA_ERR_ILLEGAL_VALUE;
    }
}

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Clear auxiliary structure progress
 */
void plasma_clear_aux_progress(int size, int value)
{
    int i;

    for (i = 0; i < size; i++)
        plasma_aux.progress[i] = value;
}

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Returns core id
 */
int plasma_self()
{
    int core;
    pthread_t thread_id;

    thread_id = pthread_self();
    for (core = 0; core < plasma_cntrl.cores_num; core++)
        if (plasma_cntrl.thread_id[core] == thread_id)
            return core;
    return PLASMA_ERR_NOT_FOUND;
}

/*////////////////////////////////////////////////////////////////////////////////////////
 *  Tune block size NB and internal block size IB
 */
int plasma_tune(PLASMA_enum func, int M, int N, int NRHS)
{
    plasma_cntrl.NB = 160;
    plasma_cntrl.IB = 40;
    /* Calculate A, B tile size and round up to cache line size */
    plasma_cntrl.NBNBSIZE = plasma_cntrl.NB * plasma_cntrl.NB * sizeof(double);
    plasma_cntrl.NBNBSIZE = roundup(plasma_cntrl.NBNBSIZE, plasma_cntrl.cache_line_size);
    plasma_cntrl.NBNBSIZE /= sizeof(double);
    /* Calculate T tile size and round up to cache line size */
    plasma_cntrl.IBNBSIZE = plasma_cntrl.IB * plasma_cntrl.NB * sizeof(double);
    plasma_cntrl.IBNBSIZE = roundup(plasma_cntrl.IBNBSIZE, plasma_cntrl.cache_line_size);
    plasma_cntrl.IBNBSIZE /= sizeof(double);
    return PLASMA_SUCCESS;
}
