#! /usr/bin/env python
# -*- coding: utf-8 -*-

###
#
# @file lapcwrapper.py
#
#  PLASMA is a software package provided by Univ. of Tennessee,
#  Univ. of California Berkeley and Univ. of Colorado Denver
#
# @version 2.4.1
# @author Julie Langou
# @author Mathieu Faverge
# @date 2010-11-15
#
###

from utils import writefile, runShellCommand, killfiles, downloader, getURLName
import sys
import os
import shutil
import framework

class Lapcwrapper(framework.Framework):
    lapcurl     = "http://icl.cs.utk.edu/projectsfiles/plasma/pubs/lapacke-3.3.0.tgz"

    """ This class takes care of the LAPACK C interface installation. """
    def __init__(self, config, plasma):
        print "\n","="*40
        print "  LAPACK C interface installation/verification"
        print "="*40

        self.config   = config
        self.downlapc = plasma.downlapc
        self.downcmd  = plasma.downcmd
        self.mangling = plasma.mangling
        self.prefix   = plasma.prefix
        self.plasma   = plasma

        if self.downlapc == 2:
            self.down_install_lapc()

        ret = self.check_lapc()
        if ret != 0:
            if self.downlapc == 1:
                self.down_install_lapc()
            else :
                print """
Please provide a working LAPACK C interface. If a LAPACK C interface is not
present on the system, the netlib LAPACK C interface can be automatically
downloaded and installed by adding the --downlapc flag.

What do you want to do ?
    - d : download automatically the netlib LAPACK C interface
    - q : quit to download and install manually a LAPACK C interface.
"""
                answer = raw_input(">[q] ")
                if answer == "d":
                    self.down_install_lapc()
                else:
                    sys.exit()

    def check_lapc(self):

        print "Checking if provided LAPACK C interface works...",
        # This function simply generates a C program
        # that contains few calls to LAPACK C interface routine and then
        # checks if compilation, linking and execution are succesful

        sys.stdout.flush()
        writefile('tmpc.c',"""
#include <lapacke.h>
int main(int argc, char*agv[]) {
  double eps;
  eps = LAPACKE_dlamch('e');
  return 0;
}
\n""")

        ccomm = self.config.cc+' '+self.config.ldflags_c+' -o tmpc.o -c tmpc.c '+self.config.lapackinc
        (output, error, retz) = runShellCommand(ccomm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C interface cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',ccomm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;

        ldflg = self.config.ldflags_c+' '+self.config.lapclib+' '+self.config.lapacklib+' '+self.config.blaslib+' -lm'
        ccomm = self.config.fc+' -o tmpc '+'tmpc.o '+self.config.lapackinc+' '+ldflg
        (output, error, retz) = runShellCommand(ccomm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C interface cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',ccomm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;

        comm = './tmpc'
        (output, error, retz) = runShellCommand(comm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C interface cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;

        killfiles(['tmpc.o','tmpc.c','tmpc'])
        print 'yes'

        return 0;


    def down_install_lapc(self):

        print """
The reference LAPACK C interface is being installed.
"""
        sys.stdout.flush()

        savecwd = os.getcwd()

        # creating the build,lib and log dirs if don't exist
        if not os.path.isdir(os.path.join(self.prefix,'lib')):
            os.mkdir(os.path.join(self.prefix,'lib'))

        if not os.path.isdir(os.path.join(self.prefix,'include')):
            os.mkdir(os.path.join(self.prefix,'include'))

        if not os.path.isdir(os.path.join(os.getcwd(),'log')):
            os.mkdir(os.path.join(os.getcwd(),'log'))

        # Check if lapacke.tgz is already present in the working dir
        # otherwise download it
        if not os.path.isfile(os.path.join(os.getcwd(),getURLName(self.lapcurl))):
            print "Downloading reference LAPACK C interface...",
            downloader(self.lapcurl,self.downcmd)
            print "done"

        # unzip and untar
        print 'Unzip and untar reference LAPACK C interface...',
        comm = 'gunzip -f lapacke-3.3.0.tgz'
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print '\n\nLAPCWRAPPER: cannot unzip lapacke.tgz'
            print 'stderr:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            sys.exit()

        comm = 'tar xf lapacke-3.3.0.tar'
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print '\n\nLAPCWRAPPER: cannot untar lapacke.tar'
            print 'stderr:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            sys.exit()
        os.remove('lapacke-3.3.0.tar')
        print 'done'

        # change to LAPACK C interface dir
        os.chdir(os.path.join(os.getcwd(),'lapacke-3.3.0'))
        sdir = os.getcwd()

        # Write Makefile.in
        writefile('make.inc', """
# make.inc (Plasma Installer)
#
#-----------------------------------------------------------------------------
# Shell
#-----------------------------------------------------------------------------
SHELL = /bin/sh

#-----------------------------------------------------------------------------
# Platform
#-----------------------------------------------------------------------------

PLAT = """+self.plat+"""

#-----------------------------------------------------------------------------
# Compilers
#-----------------------------------------------------------------------------

CC      = """+self.config.cc+""" """+self.config.ccflags+""" """+self.mangling+"""
FORTRAN = """+self.config.fc+""" """+self.config.fcflags+"""
LOADER  = $(CC)

#-----------------------------------------------------------------------------
# Flags for Compilers
#-----------------------------------------------------------------------------

OPTS  =
NOOPT =

#-----------------------------------------------------------------------------
# Archive programs and flags
#-----------------------------------------------------------------------------

ARCH      = ar
ARCHFLAGS = """+self.config.arflags+"""
RANLIB    = """+self.config.ranlib+"""

LAPACKE   = liblapacke.a
""")

        # compile and generate library
        print 'Compile and generate reference LAPACK C interface...',
        sys.stdout.flush()
        comm = self.make
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print "\n\nLAPCWRAPPER: cannot compile LAPACK C interface"
            print "stderr:\n","*"*40,"\n",comm,'\n',error,"\n","*"*40
            sys.exit()

        log = output+error

        # write the log on a file
        log = log+output+error
        fulllog = os.path.join(savecwd,'log/lapackcwrapperlog')
        writefile(fulllog, log)
        print 'Installation of reference LAPACK C interface successful.'
        print '(log is in ',fulllog,')'

        # move liblapacke.a to the lib directory
        shutil.copy('liblapacke.a',os.path.join(self.prefix,'lib/liblapacke.a'))

        # move headers to the include directory
        shutil.copy('include/lapacke.h',os.path.join(self.prefix,'include/lapacke.h'))

        # set framework variables to point to the freshly installed BLAS library
        self.config.lapclib   = '-L'+os.path.join(self.prefix,'lib')+' -llapacke'
        self.config.lapackinc = '-I'+os.path.join(self.prefix,'include')
        os.chdir(savecwd)

        # Check if the installation is successful
        self.plasma.verbose = 1
        ret = self.check_lapc()
        self.plasma.verbose = 0
        if ret != 0 :
            sys.exit()
