#! /usr/bin/env python
# -*- coding: utf-8 -*-

# -----------------------------------------
# PLASMA installer
# University of Tennessee Knoxville
# June 24, 2009
# ----------------------------------------

from utils import writefile, runShellCommand, killfiles, downloader, getURLName
import sys
import os
import shutil
import framework

class Lapcwrapper(framework.Framework):
    lapcurl     = "http://icl.cs.utk.edu/projectsfiles/plasma/pubs/lapack_cwrapper.tgz"

    """ This class takes care of the LAPACK C wrapper installation. """
    def __init__(self, config, plasma):
        print "\n","="*40
        print "  LAPACK C wrapper installation/verification"
        print "="*40

        self.config   = config
        self.downlapc = plasma.downlapc
        self.downcmd  = plasma.downcmd
        self.mangling = plasma.mangling
        self.prefix   = plasma.prefix
        self.plasma   = plasma

        ret = self.check_lapc()
        if ret != 0:
            if self.downlapc:
                self.down_install_lapc()
            else :
                print """
Please provide a working LAPACK C wrapper. If a LAPACK C wrapper is not
present on the system, the netlib LAPACK C wrapper can be automatically
downloaded and installed by adding the --downlapc flag.  

The LAPACK C wrapper is not needed in the case where testing is disabled
by means of the --notesting flag.

What do you want to do ? 
    - d : download automatically the netlib LAPACK C wrapper
    - q : quit to download and install manually a LAPACK C wrapper.
"""
                answer = raw_input(">[q] ")
                if answer == "d":
                    self.down_install_lapc()
                else:
                    sys.exit()

    def check_lapc(self):

        print "Checking if provided LAPACK C wrapper works...",
        # This function simply generates a C program
        # that contains few calls to LAPACK C wrapper routine and then
        # checks if compilation, linking and execution are succesful

        sys.stdout.flush()
        writefile('tmpc.c',"""
#include <lapack.h>
int main(int argc, char*agv[]) {
  double eps;
  eps = lapack_dlamch(lapack_eps);
  return 0;
}
\n""")

        ccomm = self.config.cc+' '+self.config.ldflags_c+' -o tmpc.o -c tmpc.c '+self.config.lapackinc
        (output, error, retz) = runShellCommand(ccomm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C wrapper cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',ccomm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;

        ldflg = self.config.ldflags_fc+' '+self.config.lapclib+' '+self.config.lapacklib+' '+self.config.blaslib+' -lm'
        ccomm = self.config.fc+' -o tmpc '+'tmpc.o '+self.config.lapackinc+' '+ldflg
        (output, error, retz) = runShellCommand(ccomm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C wrapper cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',ccomm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;

        comm = './tmpc'
        (output, error, retz) = runShellCommand(comm)
        if(retz != 0):
            if self.plasma.verbose:
                print '\n\nLAPCWRAPPER: provided LAPACK C wrapper cannot be used! aborting...'
                print 'error is:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            else:
                print "no"
            return -1;
        
        killfiles(['tmpc.o','tmpc.c','tmpc'])
        print 'yes'

        return 0;


    def down_install_lapc(self):

        print """
The reference LAPACK C wrapper is being installed.
"""
        sys.stdout.flush()

        savecwd = os.getcwd()

        # creating the build,lib and log dirs if don't exist
        if not os.path.isdir(os.path.join(self.prefix,'lib')):
            os.mkdir(os.path.join(self.prefix,'lib'))

        if not os.path.isdir(os.path.join(self.prefix,'include')):
            os.mkdir(os.path.join(self.prefix,'include'))

        if not os.path.isdir(os.path.join(os.getcwd(),'log')):
            os.mkdir(os.path.join(os.getcwd(),'log'))

        # Check if lapcwrapper.tgz is already present in the working dir
        # otherwise download it
        if not os.path.isfile(os.path.join(os.getcwd(),getURLName(self.lapcurl))):
            print "Downloading reference LAPACK C wrapper...",
            downloader(self.lapcurl,self.downcmd)
            print "done"

        # unzip and untar
        print 'Unzip and untar reference LAPACK C wrapper...',
        comm = 'gunzip -f lapack_cwrapper.tgz'
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print '\n\nLAPCWRAPPER: cannot unzip lapack_cwrapper.tgz'
            print 'stderr:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            sys.exit()

        comm = 'tar xf lapack_cwrapper.tar'
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print '\n\nLAPCWRAPPER: cannot untar lapack_cwrapper.tar'
            print 'stderr:\n','*'*40,'\n',comm,'\n',error,'\n','*'*40
            sys.exit()
        os.remove('lapack_cwrapper.tar')
        print 'done'

        # change to LAPACK C wrapper dir
        os.chdir(os.path.join(os.getcwd(),'lapack_cwrapper'))
        sdir = os.getcwd()

        # Write Makefile.in
        writefile('make.inc', """
# make.inc (Plasma Installer)
#
#-----------------------------------------------------------------------------
# Shell
#-----------------------------------------------------------------------------
SHELL = /bin/sh

#-----------------------------------------------------------------------------
# Platform
#-----------------------------------------------------------------------------

PLAT = """+self.plat+"""

#-----------------------------------------------------------------------------
# Compilers
#-----------------------------------------------------------------------------

CC      = """+self.config.cc+""" """+self.config.ccflags+""" """+self.mangling+""" -DF77_LOG=int
FORTRAN = """+self.config.fc+""" """+self.config.fcflags+"""                      
LOADER  = $(CC)

#-----------------------------------------------------------------------------
# Flags for Compilers
#-----------------------------------------------------------------------------

OPTS  = 
NOOPT = 

#-----------------------------------------------------------------------------
# Archive programs and flags
#-----------------------------------------------------------------------------

ARCH      = ar
ARCHFLAGS = """+self.config.arflags+"""
RANLIB    = """+self.config.ranlib+"""

ROOT_DIR  = """+sdir+"""
SRC_DIR   = $(ROOT_DIR)/src
LAWRAPLIB = $(ROOT_DIR)/lib/liblapack_cwrapper.a
""")

        # compile and generate library
        print 'Compile and generate reference LAPACK C wrapper...',
        sys.stdout.flush()
        comm = self.make+' library'
        (output, error, retz) = runShellCommand(comm)
        if retz:
            print "\n\nLAPCWRAPPER: cannot compile LAPACK C wrapper"
            print "stderr:\n","*"*40,"\n",comm,'\n',error,"\n","*"*40
            sys.exit()

        log = output+error

        # write the log on a file
        log = log+output+error
        fulllog = os.path.join(savecwd,'log/lapackcwrapperlog')
        writefile(fulllog, log)
        print 'Installation of reference LAPACK C wrapper successful.'
        print '(log is in ',fulllog,')'

        # move liblapcwrapper.a to the lib directory
        shutil.copy('lib/liblapack_cwrapper.a',os.path.join(self.prefix,'lib/liblapack_cwrapper.a'))

        # move headers to the include directory
        shutil.copy('include/lapack.h',os.path.join(self.prefix,'include/lapack.h'))
        shutil.copy('include/lapack_enum.h',os.path.join(self.prefix,'include/lapack_enum.h'))

        # set framework variables to point to the freshly installed BLAS library
        self.config.lapclib   = '-L'+os.path.join(self.prefix,'lib')+' -llapack_cwrapper'
        self.config.lapackinc = '-I'+os.path.join(self.prefix,'include')
        os.chdir(savecwd)

        # Check if the installation is successful
        self.plasma.verbose = 1
        ret = self.check_lapc()
        self.plasma.verbose = 0
        if ret != 0 :
            sys.exit()
