/* -*- C -*- */
/* psolve.c
 */

#include <lfc/lfci.h>

#define ASSERT(x) do{if(!(x))printf("%s(%d): assertion failed\n","prlslv.c",__LINE__);}while(0)

LFC_BEGIN_C_DECLS

static void
msg(char *m) {
  printf( "%s\n", m ); fflush( stdout );
}

static int
mtx_dwrite_urandom(int m, int n, const char *mat_fname, int seed, double mn,
		   double mx) {
  char line1[] = "LFCMATRIX"; /* this will be the magic number of the file */
  int i, j; FILE *f;
  double *x, scl = (mx - mn) / RAND_MAX;

  srand( seed );

  x = LFC_MALLOC( double, m );
  ASSERT( x );

  f = fopen( mat_fname, "wb" );
  if (! f) {
    LFC_FREE( x );
    return -1;
  }

  fprintf( f, "%s\n", line1 );
  fprintf( f, "%d %d\n", m, n );

  for (j = 0; j < n; j++) {
    for (i = 0; i < m; i++) x[i] = scl * rand() + mn;

    fwrite( x, sizeof *x, m, f );
  }

  fclose( f );

  LFC_FREE( x );

  return 0;
}	/* mtx_dwrite_urandom */

static int
psolve(int n, char *mat_fname, char *rhs_fname, char *sol_fname, int gp, int gq){
  int r, *piv; LFC_Status status;

  piv = LFC_MALLOC( int, n );
  ASSERT( piv );

  msg( "Writing matrix file..." );
  mtx_dwrite_urandom( n, n, mat_fname, 13130, 0.0, 1.0 );
  msg( "Wrote matrix file." );
  mtx_dwrite_urandom( n, 1, rhs_fname, 13939, 0.0, 1.0 );
  msg( "Wrote right-hand side copy file." );
  mtx_dwrite_urandom( n, 1, sol_fname, 13939, 0.0, 1.0 );
  msg( "Wrote righ-hand side file." );

  if (getenv("PSOLVE_TEST")) {gp = 2; gq = 2;}
  r = LFC_dgesv_file_all_x(n, 1, mat_fname, LFC_RDONLY, piv, sol_fname, 0, gp, gq, &status);
  ASSERT( LFC_SUCCESS == r );
  msg( "LFC_dgesv_file_all_x() finished." );

  if (getenv("PSOLVE_TEST")) {gp = 1; gq = 4;}
  r = LFC_dgesv_rsdl_x( n, 1, mat_fname, sol_fname, rhs_fname, gp, gq, &status );
  ASSERT( LFC_SUCCESS == r );
  msg( "LFC_dgesv_rsdl() finished." );

  LFC_FREE( piv );

  remove( mat_fname );
  remove( rhs_fname );
  remove( sol_fname );

  return 0;
}	/* psolve */

void
prlslv(int n0, int n1, int n2, char *mtxf, char *rhsf, char *solf, int nproc) {
  int gp, gq, n;

  if (! n2) n2 = 1;

  gp = 1;
  gq = nproc;
  do {
    for (n = n0; n <= n1; n += n2)
      psolve( n, mtxf, rhsf, solf, gp, gq);

    do {
      gp++;
      gq = nproc / gp;
    } while (gp <= gq && gp * gq != nproc);
  } while (gp <= gq);
}	/* prlslv */

static int
chkprlenv(int *nmin, int *nmax) {
  int i, nhsts, rv, ds; double totmem, mem, mmax; LFC_Hosts hosts;

  rv = LFC_Hosts_create( &hosts, &nhsts );
  if (LFC_FAILURE == rv) return 1;
  if (nhsts <= 1) {
    LFC_Hosts_free( &hosts );
    return 1;
  }

  LFC_Hosts_get_free_memory( &hosts, 0, &mmax );
  totmem = mmax;
  for (i = 1; i < nhsts; i++) {
    LFC_Hosts_get_free_memory( &hosts, i, &mem );
    if (mem > mmax) mmax = mem;
    totmem += mem;
  }

  LFC_Hosts_free( &hosts );

  ds = sizeof(double);

  *nmin = sqrt( mmax * 1e6 / ds ) * 1.1;
  *nmax = sqrt( totmem * 1e6 / ds );

  if (*nmin > *nmax) *nmin = *nmax;

  return 0;
}	/* chkprlenv */

static int
slen(char *s) {int l; for (l = 0; *s; s++) l++; return l;}

static void
scpy(char *dst, char *src) { for (; *src; src++, dst++) *dst = *src; *dst = *src;}

static char *
mrg(char *s1, char *s2) {
  int l1, l2; char *s;

  l1 = slen( s1 );
  l2 = slen( s2 );
  s = (char *)malloc( l1 + l2 + 1 );
  if (! s) return s;
  scpy( s, s1 );
  scpy( s + l1, s2 );
  return s;
}

static int
chkhome(char **mtxf, char **rhsf, char **solf) {
  char rhs_fname[] = "/rhs-file", sol_fname[] = "/sol-file";
  char mat_fname[] = "/../../"; char *path;

  path = getenv( "HOME" );
  if (! path) {
    printf( "You don't have the HOME environment variable set!\n" );
    return 1;
  }

  *mtxf = mrg(   "", mat_fname );
  *rhsf = mrg( path, rhs_fname );
  *solf = mrg( path, sol_fname );

  if (! *mtxf || ! *rhsf || ! *solf) return 1;

  return 0;
}	/* chkhome */

void
prlslv1(int pmin, int pmax, int pstep, int n0, int n1, int n2, char *mtxf,
	char *rhsf, char *solf) {
  int nproc;

  for (nproc = pmin; nproc <= pmax; nproc += pstep)
    prlslv( n0, n1, n2, mtxf, rhsf, solf, nproc );
}

int
main(int argc, char *argv[]) {
  int pmin, pmax, pstep, nmin, nmax, n0, n1, n2; char *mtxf, *rhsf, *solf;

  if (chkprlenv( &nmin, &nmax )) {
    msg( "No distributed memory parallel environment avaiable!" );
    exit( 77 );
    return 77;
  }

  if (chkhome( &mtxf, &rhsf, &solf )) {
    msg( "Couldn't find a place for output files!" );
    exit( 77 );
    return 77;
  }

  if (argc <= 1 || sscanf( argv[1], "%d", &pmin ) != 1 || pmin < 1) pmin = 60;
  if (argc <= 2 || sscanf( argv[2], "%d", &pmax ) != 1 || pmax < pmin) pmax = pmin;
  if (argc <= 3 || sscanf( argv[3], "%d", &pstep ) != 1 || pstep < 1) pstep = 1;
  if (argc <= 4 || sscanf( argv[4], "%d", &n0 ) != 1 || n0 < 1) n0 = 70000;
  if (argc <= 5 || sscanf( argv[5], "%d", &n1 ) != 1 || n1 < 1) n1 = n0;
  if (argc <= 6 || sscanf( argv[6], "%d", &n2 ) != 1 || n2 < 1) n2 = (n1 - n0) / 10;
  if (! n2) n2 = 1;

  prlslv1( pmin, pmax, pstep, n0, n1, n2, mtxf, rhsf, solf );

  msg( "Finished." );

  return 0;
}	/* main */

LFC_END_C_DECLS
