/* -*- C -*- */
/*
#include "pblas.h"

#include "sclapack.h"
#include "slblas.h"
*/

#include <lfc/lfci.h>

#define trans2e(t) (((t)[0]=='C'||(t)[0]=='c')?LFC_CblasConjTrans:(((t)[0]=='T'||(t)[0]=='t')?LFC_CblasTrans:LFC_CblasNoTrans))
#define side2e(t) (((t)[0]=='R'||(t)[0]=='r')?LFC_CblasRight:LFC_CblasLeft)
#define uplo2e(t) (((t)[0]=='L'||(t)[0]=='l')?LFC_CblasLower:LFC_CblasUpper)
#define diag2e(t) (((t)[0]=='N'||(t)[0]=='n')?LFC_CblasNonUnit:LFC_CblasUnit)

LFC_BEGIN_C_DECLS

void
lfc_SLsrot(int *N_, char *X_, int *incX_, char *Y_, int *incY_, char *c_, char *s_) {
  float *X = (float *)X_, *Y = (float *)Y_;
  float *c = (float *)c_, *s = (float *)s_;
  lfc_cblas_srot( *N_, X, *incX_, Y, *incY_, *c, *s );
}
void
lfc_SLsger(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  float *X = (float *)X_, *Y = (float *)Y_, *A = (float *)A_;
  float *alpha = (float *)alpha_;
  lfc_cblas_sger( LFC_CblasColMajor, *M_, *N_, *alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}

void
lfc_SLssymv(char *uplo_, int *n_, char *alpha_, char *a_, int *lda_, char *x_, int *incx_,
            char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  float *a = (float *)a_, *x = (float *)x_, *y = (float *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_ssymv( LFC_CblasColMajor, uplo, *n_, *alpha, a, *lda_, x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLssyr(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_;
  float *a = (float *)a_, *x = (float *)x_;

  uplo = uplo2e(uplo_);
  lfc_cblas_ssyr( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, a, *lda_ );
}

void
lfc_SLssyr2(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *y_,
  int *incy_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_;
  float *a = (float *)a_, *x = (float *)x_, *y = (float *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_ssyr2( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, y, *incy_, a, *lda_ );
}

void
lfc_SLssbmv(char *uplo_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  float *a = (float *)a_, *x = (float *)x_, *y = (float *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_ssbmv( LFC_CblasColMajor, uplo, *n_, *k_, *alpha, a, *lda_, x, *incx_, *beta,
                      y, *incy_ );
}

float
lfc_SLsnrm2(int *N_, char *X_, int *incX_) {
  float *X = (float *)X_;
  return lfc_cblas_snrm2( *N_, X, *incX_ );
}

float
lfc_SLsasum(int *N_, char *X_, int *incX_) {
  float *X = (float *)X_;
  return lfc_cblas_sasum( *N_, X, *incX_ );
}

int
lfc_SLisamax(int *N_, char *X_, int *incX_) {
  float *X = (float *)X_;
  return lfc_cblas_isamax( *N_, X, *incX_ ) + 1;
}

void
lfc_SLsaxpy(int *N_, char *alpha_, char *X_, int *incX_, char *Y_, int *incY_) {
  float *alpha = (float *)alpha_, *X = (float *)X_, *Y = (float *)Y_;
  lfc_cblas_saxpy( *N_, *alpha, X, *incX_, Y, *incY_ );
}

void
lfc_SLsswap(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  float *X = (float *)X_, *Y = (float *)Y_;
  lfc_cblas_sswap( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLscopy(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  float *X = (float *)X_, *Y = (float *)Y_;
  lfc_cblas_scopy( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLsscal(int *N_, char *alpha_, char *X_, int *incX_) {
  float *X = (float *)X_, *alpha = (float *)alpha_;
  lfc_cblas_sscal( *N_, *alpha, X, *incX_ );
}

/* ----- BLAS Level 2 ----- */

void
lfc_SLsgbmv(char *ta_, int *m_, int *n_, int *kl_, int *ku_, char *alpha_,
  char *a_, int *lda_, char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  float *a = (float *)a_, *x = (float *)x_, *y = (float *)y_;

  ta = trans2e(ta_);
  lfc_cblas_sgbmv( LFC_CblasColMajor, ta, *m_, *n_, *kl_, *ku_, *alpha, a, *lda_,
		   x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLsgemv(char *ta_, int *m_, int *n_, char *alpha_, char *a_, int *lda_,
	    char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  float *alpha = (float *)alpha_, *beta = (float *)beta_,
    *a = (float *)a_, *x = (float *)x_, *y = (float *)y_;

  ta = trans2e(ta_);
  lfc_cblas_sgemv( LFC_CblasColMajor, ta, *m_, *n_, *alpha, a, *lda_,
		   x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLstrmv(char *uplo_, char *ta_, char *diag_,
  int *n_, char *a_, int *lda_, char *x_, int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  float *a = (float *)a_, *x = (float *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_strmv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLstrsv(char *uplo_, char *ta_, char *diag_, int *n_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  float *a = (float *)a_, *x = (float *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_strsv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLstbsv(char *uplo_, char *ta_, char *diag_, int *n_, int *k_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  float *a = (float *)a_, *x = (float *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_stbsv( LFC_CblasColMajor, uplo, ta, diag, *n_, *k_, a, *lda_, x, *incx_ );
}

/* ----- BLAS Level 3 ----- */

void
lfc_SLsgemm(char *ta_,  char *tb_,  int *m_, int *n_, int *k_,
	    char *alpha_, char *a_, int *lda_, char *b_, int *ldb_,
            char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta, tb;
  float *alpha = (float *)alpha_, *beta = (float *)beta_,
    *a = (float *)a_, *b = (float *)b_, *c = (float *)c_;

  ta = trans2e(ta_);
  tb = trans2e(tb_);
  lfc_cblas_sgemm( LFC_CblasColMajor, ta, tb, *m_, *n_, *k_, *alpha, a, *lda_,
		   b, *ldb_, *beta, c, *ldc_ );
}
void
lfc_SLstrmm(char *side_,  char *uplo_, char *ta_, char *diag_,
  int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_, int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  float *alpha = (float *)alpha_, *a = (float *)a_, *b = (float *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_strmm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, *alpha,
		   a, *lda_, b, *ldb_ );
}
void
lfc_SLstrsm(char *side_,  char *uplo_, char *ta_, char *diag_,
            int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_,
            int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  float *alpha = (float *)alpha_, *a = (float *)a_, *b = (float *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_strsm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, *alpha,
		   a, *lda_, b, *ldb_ );
}

void
lfc_SLssyrk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_, *beta = (float *)beta_, *a = (float *)a_;
  float *c = (float *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_ssyrk( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_,
		   *beta, c, *ldc_ );
}

void
lfc_SLssymm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  float *a = (float *)a_, *b = (float *)b_, *c = (float *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_ssymm( LFC_CblasColMajor, side, uplo, *m_, *n_, *alpha, a, *lda_,
		   b, *ldb_, *beta, c, *ldc_ );
}

void
lfc_SLssyr2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  float *a = (float *)a_, *b = (float *)b_, *c = (float *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_ssyr2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_, b, *ldb_,
		       *beta, c, *ldc_ );
}

void
lfc_SLdrot(int *N_, char *X_, int *incX_, char *Y_, int *incY_, char *c_, char *s_) {
  double *X = (double *)X_, *Y = (double *)Y_;
  double *c = (double *)c_, *s = (double *)s_;
  lfc_cblas_drot( *N_, X, *incX_, Y, *incY_, *c, *s );
}
void
lfc_SLdger(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  double *X = (double *)X_, *Y = (double *)Y_, *A = (double *)A_;
  double *alpha = (double *)alpha_;
  lfc_cblas_dger( LFC_CblasColMajor, *M_, *N_, *alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}

void
lfc_SLdsymv(char *uplo_, int *n_, char *alpha_, char *a_, int *lda_, char *x_, int *incx_,
            char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  double *a = (double *)a_, *x = (double *)x_, *y = (double *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_dsymv( LFC_CblasColMajor, uplo, *n_, *alpha, a, *lda_, x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLdsyr(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_;
  double *a = (double *)a_, *x = (double *)x_;

  uplo = uplo2e(uplo_);
  lfc_cblas_dsyr( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, a, *lda_ );
}

void
lfc_SLdsyr2(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *y_,
  int *incy_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_;
  double *a = (double *)a_, *x = (double *)x_, *y = (double *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_dsyr2( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, y, *incy_, a, *lda_ );
}

void
lfc_SLdsbmv(char *uplo_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  double *a = (double *)a_, *x = (double *)x_, *y = (double *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_dsbmv( LFC_CblasColMajor, uplo, *n_, *k_, *alpha, a, *lda_, x, *incx_, *beta,
                      y, *incy_ );
}

double
lfc_SLdnrm2(int *N_, char *X_, int *incX_) {
  double *X = (double *)X_;
  return lfc_cblas_dnrm2( *N_, X, *incX_ );
}

double
lfc_SLdasum(int *N_, char *X_, int *incX_) {
  double *X = (double *)X_;
  return lfc_cblas_dasum( *N_, X, *incX_ );
}

int
lfc_SLidamax(int *N_, char *X_, int *incX_) {
  double *X = (double *)X_;
  return lfc_cblas_idamax( *N_, X, *incX_ ) + 1;
}

void
lfc_SLdaxpy(int *N_, char *alpha_, char *X_, int *incX_, char *Y_, int *incY_) {
  double *alpha = (double *)alpha_, *X = (double *)X_, *Y = (double *)Y_;
  lfc_cblas_daxpy( *N_, *alpha, X, *incX_, Y, *incY_ );
}

void
lfc_SLdswap(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  double *X = (double *)X_, *Y = (double *)Y_;
  lfc_cblas_dswap( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLdcopy(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  double *X = (double *)X_, *Y = (double *)Y_;
  lfc_cblas_dcopy( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLdscal(int *N_, char *alpha_, char *X_, int *incX_) {
  double *X = (double *)X_, *alpha = (double *)alpha_;
  lfc_cblas_dscal( *N_, *alpha, X, *incX_ );
}

/* ----- BLAS Level 2 ----- */

void
lfc_SLdgbmv(char *ta_, int *m_, int *n_, int *kl_, int *ku_, char *alpha_,
  char *a_, int *lda_, char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  double *a = (double *)a_, *x = (double *)x_, *y = (double *)y_;

  ta = trans2e(ta_);
  lfc_cblas_dgbmv( LFC_CblasColMajor, ta, *m_, *n_, *kl_, *ku_, *alpha, a, *lda_,
		   x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLdgemv(char *ta_, int *m_, int *n_, char *alpha_, char *a_, int *lda_,
	    char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  double *alpha = (double *)alpha_, *beta = (double *)beta_,
    *a = (double *)a_, *x = (double *)x_, *y = (double *)y_;

  ta = trans2e(ta_);
  lfc_cblas_dgemv( LFC_CblasColMajor, ta, *m_, *n_, *alpha, a, *lda_,
		   x, *incx_, *beta, y, *incy_ );
}

void
lfc_SLdtrmv(char *uplo_, char *ta_, char *diag_,
  int *n_, char *a_, int *lda_, char *x_, int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  double *a = (double *)a_, *x = (double *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_dtrmv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLdtrsv(char *uplo_, char *ta_, char *diag_, int *n_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  double *a = (double *)a_, *x = (double *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_dtrsv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLdtbsv(char *uplo_, char *ta_, char *diag_, int *n_, int *k_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  double *a = (double *)a_, *x = (double *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_dtbsv( LFC_CblasColMajor, uplo, ta, diag, *n_, *k_, a, *lda_, x, *incx_ );
}

/* ----- BLAS Level 3 ----- */

void
lfc_SLdgemm(char *ta_,  char *tb_,  int *m_, int *n_, int *k_,
	    char *alpha_, char *a_, int *lda_, char *b_, int *ldb_,
            char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta, tb;
  double *alpha = (double *)alpha_, *beta = (double *)beta_,
    *a = (double *)a_, *b = (double *)b_, *c = (double *)c_;

  ta = trans2e(ta_);
  tb = trans2e(tb_);
  lfc_cblas_dgemm( LFC_CblasColMajor, ta, tb, *m_, *n_, *k_, *alpha, a, *lda_,
		   b, *ldb_, *beta, c, *ldc_ );
}
void
lfc_SLdtrmm(char *side_,  char *uplo_, char *ta_, char *diag_,
  int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_, int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  double *alpha = (double *)alpha_, *a = (double *)a_, *b = (double *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_dtrmm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, *alpha,
		   a, *lda_, b, *ldb_ );
}
void
lfc_SLdtrsm(char *side_,  char *uplo_, char *ta_, char *diag_,
            int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_,
            int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  double *alpha = (double *)alpha_, *a = (double *)a_, *b = (double *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_dtrsm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, *alpha,
		   a, *lda_, b, *ldb_ );
}

void
lfc_SLdsyrk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_, *beta = (double *)beta_, *a = (double *)a_;
  double *c = (double *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_dsyrk( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_,
		   *beta, c, *ldc_ );
}

void
lfc_SLdsymm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  double *a = (double *)a_, *b = (double *)b_, *c = (double *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_dsymm( LFC_CblasColMajor, side, uplo, *m_, *n_, *alpha, a, *lda_,
		   b, *ldb_, *beta, c, *ldc_ );
}

void
lfc_SLdsyr2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  double *a = (double *)a_, *b = (double *)b_, *c = (double *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_dsyr2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_, b, *ldb_,
		       *beta, c, *ldc_ );
}

void
lfc_SLcgerc(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  void *X = (void *)X_, *Y = (void *)Y_, *A = (void *)A_;
  void *alpha = (void *)alpha_;
  lfc_cblas_cgerc( LFC_CblasColMajor, *M_, *N_, alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}
void
lfc_SLcgeru(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  void *X = (void *)X_, *Y = (void *)Y_, *A = (void *)A_;
  void *alpha = (void *)alpha_;
  lfc_cblas_cgeru( LFC_CblasColMajor, *M_, *N_, alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}

void
lfc_SLcsscal(int *N_, char *alpha_, char *X_, int *incX_) {
  void *X = (void *)X_;
  float *alpha = (float *)alpha_;
  lfc_cblas_csscal( *N_, *alpha, X, *incX_ );
}

void
lfc_SLcher(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_;
  void *a = (void *)a_, *x = (void *)x_;

  uplo = uplo2e(uplo_);
  lfc_cblas_cher( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, a, *lda_ );
}

void
lfc_SLcher2(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *y_,
  int *incy_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_cher2( LFC_CblasColMajor, uplo, *n_, alpha, x, *incx_, y, *incy_, a, *lda_ );
}

void
lfc_SLcherk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  float *alpha = (float *)alpha_, *beta = (float *)beta_;
  void *a = (void *)a_, *c = (void *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_cherk( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_,
		   *beta, c, *ldc_ );
}

void
lfc_SLcher2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_;
  float *beta = (float *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_cher2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_, b, *ldb_,
		       *beta, c, *ldc_ );
}

void
lfc_SLchemv(char *uplo_, int *n_, char *alpha_, char *a_, int *lda_, char *x_, int *incx_,
            char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_chemv( LFC_CblasColMajor, uplo, *n_, alpha, a, *lda_, x, *incx_, beta, y, *incy_ );
}

void
lfc_SLchemm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_chemm( LFC_CblasColMajor, side, uplo, *m_, *n_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}

void
lfc_SLchbmv(char *uplo_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_chbmv( LFC_CblasColMajor, uplo, *n_, *k_, alpha, a, *lda_, x, *incx_, beta,
                      y, *incy_ );
}

float
lfc_SLscnrm2(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_scnrm2( *N_, X, *incX_ );
}

float
lfc_SLscasum(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_scasum( *N_, X, *incX_ );
}

int
lfc_SLicamax(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_icamax( *N_, X, *incX_ ) + 1;
}

void
lfc_SLcaxpy(int *N_, char *alpha_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *alpha = (void *)alpha_, *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_caxpy( *N_, alpha, X, *incX_, Y, *incY_ );
}

void
lfc_SLcswap(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_cswap( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLccopy(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_ccopy( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLcscal(int *N_, char *alpha_, char *X_, int *incX_) {
  void *X = (void *)X_, *alpha = (void *)alpha_;
  lfc_cblas_cscal( *N_, alpha, X, *incX_ );
}

/* ----- BLAS Level 2 ----- */

void
lfc_SLcgbmv(char *ta_, int *m_, int *n_, int *kl_, int *ku_, char *alpha_,
  char *a_, int *lda_, char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  ta = trans2e(ta_);
  lfc_cblas_cgbmv( LFC_CblasColMajor, ta, *m_, *n_, *kl_, *ku_, alpha, a, *lda_,
		   x, *incx_, beta, y, *incy_ );
}

void
lfc_SLcgemv(char *ta_, int *m_, int *n_, char *alpha_, char *a_, int *lda_,
	    char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_,
    *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  ta = trans2e(ta_);
  lfc_cblas_cgemv( LFC_CblasColMajor, ta, *m_, *n_, alpha, a, *lda_,
		   x, *incx_, beta, y, *incy_ );
}

void
lfc_SLctrmv(char *uplo_, char *ta_, char *diag_,
  int *n_, char *a_, int *lda_, char *x_, int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ctrmv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLctrsv(char *uplo_, char *ta_, char *diag_, int *n_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ctrsv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLctbsv(char *uplo_, char *ta_, char *diag_, int *n_, int *k_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ctbsv( LFC_CblasColMajor, uplo, ta, diag, *n_, *k_, a, *lda_, x, *incx_ );
}

/* ----- BLAS Level 3 ----- */

void
lfc_SLcgemm(char *ta_,  char *tb_,  int *m_, int *n_, int *k_,
	    char *alpha_, char *a_, int *lda_, char *b_, int *ldb_,
            char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta, tb;
  void *alpha = (void *)alpha_, *beta = (void *)beta_,
    *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  ta = trans2e(ta_);
  tb = trans2e(tb_);
  lfc_cblas_cgemm( LFC_CblasColMajor, ta, tb, *m_, *n_, *k_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}
void
lfc_SLctrmm(char *side_,  char *uplo_, char *ta_, char *diag_,
  int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_, int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *alpha = (void *)alpha_, *a = (void *)a_, *b = (void *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ctrmm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, alpha,
		   a, *lda_, b, *ldb_ );
}
void
lfc_SLctrsm(char *side_,  char *uplo_, char *ta_, char *diag_,
            int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_,
            int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *alpha = (void *)alpha_, *a = (void *)a_, *b = (void *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ctrsm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, alpha,
		   a, *lda_, b, *ldb_ );
}

void
lfc_SLcsyrk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_, *a = (void *)a_;
  void *c = (void *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_csyrk( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_,
		   beta, c, *ldc_ );
}

void
lfc_SLcsymm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_csymm( LFC_CblasColMajor, side, uplo, *m_, *n_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}

void
lfc_SLcsyr2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_csyr2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_, b, *ldb_,
		       beta, c, *ldc_ );
}

void
lfc_SLzgerc(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  void *X = (void *)X_, *Y = (void *)Y_, *A = (void *)A_;
  void *alpha = (void *)alpha_;
  lfc_cblas_zgerc( LFC_CblasColMajor, *M_, *N_, alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}
void
lfc_SLzgeru(int *M_, int *N_, char *alpha_, char *X_, int *incX_, char *Y_,
	   int *incY_, char *A_, int *ldA_) {
  void *X = (void *)X_, *Y = (void *)Y_, *A = (void *)A_;
  void *alpha = (void *)alpha_;
  lfc_cblas_zgeru( LFC_CblasColMajor, *M_, *N_, alpha, X, *incX_, Y, *incY_, A,
		  *ldA_ );
}

void
lfc_SLzdscal(int *N_, char *alpha_, char *X_, int *incX_) {
  void *X = (void *)X_;
  double *alpha = (double *)alpha_;
  lfc_cblas_zdscal( *N_, *alpha, X, *incX_ );
}

void
lfc_SLzher(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_;
  void *a = (void *)a_, *x = (void *)x_;

  uplo = uplo2e(uplo_);
  lfc_cblas_zher( LFC_CblasColMajor, uplo, *n_, *alpha, x, *incx_, a, *lda_ );
}

void
lfc_SLzher2(char *uplo_, int *n_, char *alpha_, char *x_, int *incx_, char *y_,
  int *incy_, char *a_, int *lda_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_zher2( LFC_CblasColMajor, uplo, *n_, alpha, x, *incx_, y, *incy_, a, *lda_ );
}

void
lfc_SLzherk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  double *alpha = (double *)alpha_, *beta = (double *)beta_;
  void *a = (void *)a_, *c = (void *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_zherk( LFC_CblasColMajor, uplo, ta, *n_, *k_, *alpha, a, *lda_,
		   *beta, c, *ldc_ );
}

void
lfc_SLzher2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_;
  double *beta = (double *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_zher2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_, b, *ldb_,
		       *beta, c, *ldc_ );
}

void
lfc_SLzhemv(char *uplo_, int *n_, char *alpha_, char *a_, int *lda_, char *x_, int *incx_,
            char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_zhemv( LFC_CblasColMajor, uplo, *n_, alpha, a, *lda_, x, *incx_, beta, y, *incy_ );
}

void
lfc_SLzhemm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_zhemm( LFC_CblasColMajor, side, uplo, *m_, *n_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}

void
lfc_SLzhbmv(char *uplo_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  uplo = uplo2e(uplo_);
  lfc_cblas_zhbmv( LFC_CblasColMajor, uplo, *n_, *k_, alpha, a, *lda_, x, *incx_, beta,
                      y, *incy_ );
}

double
lfc_SLdznrm2(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_dznrm2( *N_, X, *incX_ );
}

double
lfc_SLdzasum(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_dzasum( *N_, X, *incX_ );
}

int
lfc_SLizamax(int *N_, char *X_, int *incX_) {
  void *X = (void *)X_;
  return lfc_cblas_izamax( *N_, X, *incX_ ) + 1;
}

void
lfc_SLzaxpy(int *N_, char *alpha_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *alpha = (void *)alpha_, *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_zaxpy( *N_, alpha, X, *incX_, Y, *incY_ );
}

void
lfc_SLzswap(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_zswap( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLzcopy(int *N_, char *X_, int *incX_, char *Y_, int *incY_) {
  void *X = (void *)X_, *Y = (void *)Y_;
  lfc_cblas_zcopy( *N_, X, *incX_, Y, *incY_ );
}

void
lfc_SLzscal(int *N_, char *alpha_, char *X_, int *incX_) {
  void *X = (void *)X_, *alpha = (void *)alpha_;
  lfc_cblas_zscal( *N_, alpha, X, *incX_ );
}

/* ----- BLAS Level 2 ----- */

void
lfc_SLzgbmv(char *ta_, int *m_, int *n_, int *kl_, int *ku_, char *alpha_,
  char *a_, int *lda_, char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  ta = trans2e(ta_);
  lfc_cblas_zgbmv( LFC_CblasColMajor, ta, *m_, *n_, *kl_, *ku_, alpha, a, *lda_,
		   x, *incx_, beta, y, *incy_ );
}

void
lfc_SLzgemv(char *ta_, int *m_, int *n_, char *alpha_, char *a_, int *lda_,
	    char *x_, int *incx_, char *beta_, char *y_, int *incy_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_,
    *a = (void *)a_, *x = (void *)x_, *y = (void *)y_;

  ta = trans2e(ta_);
  lfc_cblas_zgemv( LFC_CblasColMajor, ta, *m_, *n_, alpha, a, *lda_,
		   x, *incx_, beta, y, *incy_ );
}

void
lfc_SLztrmv(char *uplo_, char *ta_, char *diag_,
  int *n_, char *a_, int *lda_, char *x_, int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ztrmv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLztrsv(char *uplo_, char *ta_, char *diag_, int *n_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ztrsv( LFC_CblasColMajor, uplo, ta, diag, *n_, a, *lda_, x, *incx_ );
}

void
lfc_SLztbsv(char *uplo_, char *ta_, char *diag_, int *n_, int *k_, char *a_, int *lda_, char *x_,
  int *incx_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *a = (void *)a_, *x = (void *)x_;

  ta   = trans2e(ta_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ztbsv( LFC_CblasColMajor, uplo, ta, diag, *n_, *k_, a, *lda_, x, *incx_ );
}

/* ----- BLAS Level 3 ----- */

void
lfc_SLzgemm(char *ta_,  char *tb_,  int *m_, int *n_, int *k_,
	    char *alpha_, char *a_, int *lda_, char *b_, int *ldb_,
            char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta, tb;
  void *alpha = (void *)alpha_, *beta = (void *)beta_,
    *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  ta = trans2e(ta_);
  tb = trans2e(tb_);
  lfc_cblas_zgemm( LFC_CblasColMajor, ta, tb, *m_, *n_, *k_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}
void
lfc_SLztrmm(char *side_,  char *uplo_, char *ta_, char *diag_,
  int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_, int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *alpha = (void *)alpha_, *a = (void *)a_, *b = (void *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ztrmm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, alpha,
		   a, *lda_, b, *ldb_ );
}
void
lfc_SLztrsm(char *side_,  char *uplo_, char *ta_, char *diag_,
            int *m_, int *n_, char *alpha_, char *a_, int *lda_, char *b_,
            int *ldb_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_SIDE side;
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_DIAG diag;
  void *alpha = (void *)alpha_, *a = (void *)a_, *b = (void *)b_;

  ta   = trans2e(ta_);
  side = side2e(side_);
  uplo = uplo2e(uplo_);
  diag = diag2e(diag_);
  lfc_cblas_ztrsm( LFC_CblasColMajor, side, uplo, ta, diag, *m_, *n_, alpha,
		   a, *lda_, b, *ldb_ );
}

void
lfc_SLzsyrk(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_,
	    char *a_, int *lda_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_TRANSPOSE ta;
  enum LFC_CBLAS_UPLO uplo;
  void *alpha = (void *)alpha_, *beta = (void *)beta_, *a = (void *)a_;
  void *c = (void *)c_;
  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_zsyrk( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_,
		   beta, c, *ldc_ );
}

void
lfc_SLzsymm(char *side_, char *uplo_, int *m_, int *n_, char *alpha_,
  char *a_, int *lda_, char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_SIDE side;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  side = side2e(side_);
  lfc_cblas_zsymm( LFC_CblasColMajor, side, uplo, *m_, *n_, alpha, a, *lda_,
		   b, *ldb_, beta, c, *ldc_ );
}

void
lfc_SLzsyr2k(char *uplo_, char *ta_, int *n_, int *k_, char *alpha_, char *a_, int *lda_,
  char *b_, int *ldb_, char *beta_, char *c_, int *ldc_) {
  enum LFC_CBLAS_UPLO uplo;
  enum LFC_CBLAS_TRANSPOSE ta;
  void *alpha = (void *)alpha_, *beta = (void *)beta_;
  void *a = (void *)a_, *b = (void *)b_, *c = (void *)c_;

  uplo = uplo2e(uplo_);
  ta   = trans2e(ta_);
  lfc_cblas_zsyr2k( LFC_CblasColMajor, uplo, ta, *n_, *k_, alpha, a, *lda_, b, *ldb_,
		       beta, c, *ldc_ );
}
#define VOID void
#define r_cnjg lfc_r_cnjg
#define d_cnjg lfc_d_cnjg
/* lfc_SLddot.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


doublereal lfc_SLddot(integer *n, doublereal *dx, integer *incx, doublereal *dy, 
	integer *incy)
{
    /* System generated locals */
    integer i__1;
    doublereal ret_val;

    /* Local variables */
    integer i__, m, ix, iy, mp1;
    doublereal dtemp;


/*     forms thelfc_SL dot product of two vectors. */
/*     uses unrolled loops for increments equal to one. */
/*     jack dongarra, linpack, 3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */


    /* Parameter adjustments */
    --dy;
    --dx;

    /* Function Body */
    ret_val = 0.;
    dtemp = 0.;
    if (*n <= 0) {
	return ret_val;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	dtemp += dx[ix] * dy[iy];
	ix += *incx;
	iy += *incy;
/* L10: */
    }
    ret_val = dtemp;
    return ret_val;

/*        code for both increments equal to 1 */


/*        clean-up loop */

L20:
    m = *n % 5;
    if (m == 0) {
	goto L40;
    }
    i__1 = m;
    for (i__ = 1; i__ <= i__1; ++i__) {
	dtemp += dx[i__] * dy[i__];
/* L30: */
    }
    if (*n < 5) {
	goto L60;
    }
L40:
    mp1 = m + 1;
    i__1 = *n;
    for (i__ = mp1; i__ <= i__1; i__ += 5) {
	dtemp = dtemp + dx[i__] * dy[i__] + dx[i__ + 1] * dy[i__ + 1] + dx[
		i__ + 2] * dy[i__ + 2] + dx[i__ + 3] * dy[i__ + 3] + dx[i__ + 
		4] * dy[i__ + 4];
/* L50: */
    }
L60:
    ret_val = dtemp;
    return ret_val;
} /* lfc_SLddot_ */

/* lfc_SLsdot.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


real lfc_SLsdot(integer *n, real *sx, integer *incx, real *sy, integer *incy)
{
    /* System generated locals */
    integer i__1;
    real ret_val;

    /* Local variables */
    integer i__, m, ix, iy, mp1;
    real stemp;


/*     forms thelfc_SL dot product of two vectors. */
/*     uses unrolled loops for increments equal to one. */
/*     jack dongarra, linpack, 3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */


    /* Parameter adjustments */
    --sy;
    --sx;

    /* Function Body */
    stemp = 0.f;
    ret_val = 0.f;
    if (*n <= 0) {
	return ret_val;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	stemp += sx[ix] * sy[iy];
	ix += *incx;
	iy += *incy;
/* L10: */
    }
    ret_val = stemp;
    return ret_val;

/*        code for both increments equal to 1 */


/*        clean-up loop */

L20:
    m = *n % 5;
    if (m == 0) {
	goto L40;
    }
    i__1 = m;
    for (i__ = 1; i__ <= i__1; ++i__) {
	stemp += sx[i__] * sy[i__];
/* L30: */
    }
    if (*n < 5) {
	goto L60;
    }
L40:
    mp1 = m + 1;
    i__1 = *n;
    for (i__ = mp1; i__ <= i__1; i__ += 5) {
	stemp = stemp + sx[i__] * sy[i__] + sx[i__ + 1] * sy[i__ + 1] + sx[
		i__ + 2] * sy[i__ + 2] + sx[i__ + 3] * sy[i__ + 3] + sx[i__ + 
		4] * sy[i__ + 4];
/* L50: */
    }
L60:
    ret_val = stemp;
    return ret_val;
} /* lfc_SLsdot_ */

/* lfc_SLcdotc.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


/* Complex */ VOID lfc_SLcdotc(complex * ret_val, integer *n, complex *cx, integer 
	*incx, complex *cy, integer *incy)
{
    /* System generated locals */
    integer i__1, i__2;
    complex q__1, q__2, q__3;

    /* Builtin functions */
    void r_cnjg(complex *, complex *);

    /* Local variables */
    integer i__, ix, iy;
    complex ctemp;


/*     forms thelfc_SL dot product of two vectors, conjugating the first */
/*     vector. */
/*     jack dongarra, linpack,  3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */


    /* Parameter adjustments */
    --cy;
    --cx;

    /* Function Body */
    ctemp.r = 0.f, ctemp.i = 0.f;
     ret_val->r = 0.f,  ret_val->i = 0.f;
    if (*n <= 0) {
	return ;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	r_cnjg(&q__3, &cx[ix]);
	i__2 = iy;
	q__2.r = q__3.r * cy[i__2].r - q__3.i * cy[i__2].i, q__2.i = q__3.r * 
		cy[i__2].i + q__3.i * cy[i__2].r;
	q__1.r = ctemp.r + q__2.r, q__1.i = ctemp.i + q__2.i;
	ctemp.r = q__1.r, ctemp.i = q__1.i;
	ix += *incx;
	iy += *incy;
/* L10: */
    }
     ret_val->r = ctemp.r,  ret_val->i = ctemp.i;
    return ;

/*        code for both increments equal to 1 */

L20:
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	r_cnjg(&q__3, &cx[i__]);
	i__2 = i__;
	q__2.r = q__3.r * cy[i__2].r - q__3.i * cy[i__2].i, q__2.i = q__3.r * 
		cy[i__2].i + q__3.i * cy[i__2].r;
	q__1.r = ctemp.r + q__2.r, q__1.i = ctemp.i + q__2.i;
	ctemp.r = q__1.r, ctemp.i = q__1.i;
/* L30: */
    }
     ret_val->r = ctemp.r,  ret_val->i = ctemp.i;
    return ;
} /* lfc_SLcdotc_ */

/* lfc_SLcdotu.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


/* Complex */ VOID lfc_SLcdotu(complex * ret_val, integer *n, complex *cx, integer 
	*incx, complex *cy, integer *incy)
{
    /* System generated locals */
    integer i__1, i__2, i__3;
    complex q__1, q__2;

    /* Local variables */
    integer i__, ix, iy;
    complex ctemp;


/*     forms thelfc_SL dot product of two vectors. */
/*     jack dongarra, linpack, 3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */


    /* Parameter adjustments */
    --cy;
    --cx;

    /* Function Body */
    ctemp.r = 0.f, ctemp.i = 0.f;
     ret_val->r = 0.f,  ret_val->i = 0.f;
    if (*n <= 0) {
	return ;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	i__2 = ix;
	i__3 = iy;
	q__2.r = cx[i__2].r * cy[i__3].r - cx[i__2].i * cy[i__3].i, q__2.i = 
		cx[i__2].r * cy[i__3].i + cx[i__2].i * cy[i__3].r;
	q__1.r = ctemp.r + q__2.r, q__1.i = ctemp.i + q__2.i;
	ctemp.r = q__1.r, ctemp.i = q__1.i;
	ix += *incx;
	iy += *incy;
/* L10: */
    }
     ret_val->r = ctemp.r,  ret_val->i = ctemp.i;
    return ;

/*        code for both increments equal to 1 */

L20:
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	i__2 = i__;
	i__3 = i__;
	q__2.r = cx[i__2].r * cy[i__3].r - cx[i__2].i * cy[i__3].i, q__2.i = 
		cx[i__2].r * cy[i__3].i + cx[i__2].i * cy[i__3].r;
	q__1.r = ctemp.r + q__2.r, q__1.i = ctemp.i + q__2.i;
	ctemp.r = q__1.r, ctemp.i = q__1.i;
/* L30: */
    }
     ret_val->r = ctemp.r,  ret_val->i = ctemp.i;
    return ;
} /* lfc_SLcdotu_ */

/* lfc_SLzdotc.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


/* Double Complex */ VOID lfc_SLzdotc(doublecomplex * ret_val, integer *n, 
	doublecomplex *zx, integer *incx, doublecomplex *zy, integer *incy)
{
    /* System generated locals */
    integer i__1, i__2;
    doublecomplex z__1, z__2, z__3;

    /* Builtin functions */
    void d_cnjg(doublecomplex *, doublecomplex *);

    /* Local variables */
    integer i__, ix, iy;
    doublecomplex ztemp;


/*     forms thelfc_SL dot product of a vector. */
/*     jack dongarra, 3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */

    /* Parameter adjustments */
    --zy;
    --zx;

    /* Function Body */
    ztemp.r = 0., ztemp.i = 0.;
     ret_val->r = 0.,  ret_val->i = 0.;
    if (*n <= 0) {
	return ;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	d_cnjg(&z__3, &zx[ix]);
	i__2 = iy;
	z__2.r = z__3.r * zy[i__2].r - z__3.i * zy[i__2].i, z__2.i = z__3.r * 
		zy[i__2].i + z__3.i * zy[i__2].r;
	z__1.r = ztemp.r + z__2.r, z__1.i = ztemp.i + z__2.i;
	ztemp.r = z__1.r, ztemp.i = z__1.i;
	ix += *incx;
	iy += *incy;
/* L10: */
    }
     ret_val->r = ztemp.r,  ret_val->i = ztemp.i;
    return ;

/*        code for both increments equal to 1 */

L20:
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	d_cnjg(&z__3, &zx[i__]);
	i__2 = i__;
	z__2.r = z__3.r * zy[i__2].r - z__3.i * zy[i__2].i, z__2.i = z__3.r * 
		zy[i__2].i + z__3.i * zy[i__2].r;
	z__1.r = ztemp.r + z__2.r, z__1.i = ztemp.i + z__2.i;
	ztemp.r = z__1.r, ztemp.i = z__1.i;
/* L30: */
    }
     ret_val->r = ztemp.r,  ret_val->i = ztemp.i;
    return ;
} /* lfc_SLzdotc_ */

/* lfc_SLzdotu.f -- translated by f2c (version 20031025).
   You must link the resulting object file with libf2c:
	on Microsoft Windows system, link with libf2c.lib;
	on Linux or Unix systems, link with .../path/to/libf2c.a -lm
	or, if you install libf2c.a in a standard place, with -lf2c -lm
	-- in that order, at the end of the command line, as in
		cc *.o -lf2c -lm
	Source for libf2c is in /netlib/f2c/libf2c.zip, e.g.,

		http://www.netlib.org/f2c/libf2c.zip
*/


/* Double Complex */ VOID lfc_SLzdotu(doublecomplex * ret_val, integer *n, 
	doublecomplex *zx, integer *incx, doublecomplex *zy, integer *incy)
{
    /* System generated locals */
    integer i__1, i__2, i__3;
    doublecomplex z__1, z__2;

    /* Local variables */
    integer i__, ix, iy;
    doublecomplex ztemp;


/*     forms thelfc_SL dot product of two vectors. */
/*     jack dongarra, 3/11/78. */
/*     modified 12/3/93, array(1) declarations changed to array(*) */

    /* Parameter adjustments */
    --zy;
    --zx;

    /* Function Body */
    ztemp.r = 0., ztemp.i = 0.;
     ret_val->r = 0.,  ret_val->i = 0.;
    if (*n <= 0) {
	return ;
    }
    if (*incx == 1 && *incy == 1) {
	goto L20;
    }

/*        code for unequal increments or equal increments */
/*          not equal to 1 */

    ix = 1;
    iy = 1;
    if (*incx < 0) {
	ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
	iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	i__2 = ix;
	i__3 = iy;
	z__2.r = zx[i__2].r * zy[i__3].r - zx[i__2].i * zy[i__3].i, z__2.i = 
		zx[i__2].r * zy[i__3].i + zx[i__2].i * zy[i__3].r;
	z__1.r = ztemp.r + z__2.r, z__1.i = ztemp.i + z__2.i;
	ztemp.r = z__1.r, ztemp.i = z__1.i;
	ix += *incx;
	iy += *incy;
/* L10: */
    }
     ret_val->r = ztemp.r,  ret_val->i = ztemp.i;
    return ;

/*        code for both increments equal to 1 */

L20:
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
	i__2 = i__;
	i__3 = i__;
	z__2.r = zx[i__2].r * zy[i__3].r - zx[i__2].i * zy[i__3].i, z__2.i = 
		zx[i__2].r * zy[i__3].i + zx[i__2].i * zy[i__3].r;
	z__1.r = ztemp.r + z__2.r, z__1.i = ztemp.i + z__2.i;
	ztemp.r = z__1.r, ztemp.i = z__1.i;
/* L30: */
    }
     ret_val->r = ztemp.r,  ret_val->i = ztemp.i;
    return ;
} /* lfc_SLzdotu_ */


LFC_END_C_DECLS
